<?php
/**
 * Plugin Name: Celeb-Right
 * Description: Configurable holiday/season messages with snow/leaves/flags, custom dates, locations, and JSON import/export.
 * Version:     4.0.0
 * Author:      Ace Kupeli
 * License:     GPL2+
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! class_exists( 'AHS_Holiday_Effects' ) ) {

    class AHS_Holiday_Effects {

        private static $instance = null;
        private $option_name     = 'ahs_settings';
        private $settings        = array();
        private $today_holiday   = null;
        private $banner_rendered = false; 

        public static function get_instance() {
            if ( null === self::$instance ) {
                self::$instance = new self();
            }
            return self::$instance;
        }

        private function __construct() {
            $this->settings = $this->get_settings();

            add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
            add_action( 'admin_init', array( $this, 'register_settings' ) );
			add_action( 'admin_enqueue_scripts', array( $this, 'admin_assets' ) );    
            add_action( 'init', array( $this, 'determine_today_holiday' ) );
            add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );  
            add_action( 'wp_body_open', array( $this, 'render_banner' ) );
            add_action( 'wp_footer', array( $this, 'render_banner' ) );
        }
		
		public function admin_assets( $hook ) {
			if ( strpos( $hook, 'ahs-holiday-effects' ) === false ) {
				return;
			}

			wp_enqueue_style( 'wp-color-picker' );
			wp_enqueue_script( 'wp-color-picker' );
			wp_enqueue_media();
		} 
		
        private function get_default_holidays() {
            $year          = date( 'Y' );
            $mlk_ts        = strtotime( "third monday of january $year" );
            $presidents_ts = strtotime( "third monday of february $year" );
            $memorial_ts   = strtotime( "last monday of may $year" );
            $labor_ts      = strtotime( "first monday of september $year" );
            $thank_ts      = strtotime( "fourth thursday of november $year" );
            $base = array(
                'bg_image'      => '',
                'bg_repeat'     => 'no-repeat',
                'bg_position'   => 'center top',
                'bg_size'       => 'cover',
                'banner_height' => '',
				'text_h_align'  => 'center', 
				'text_v_align'  => 'middle',
				'text_offset_x' => '',    
				'text_offset_y' => '', 
				'font_family'   => '',
                'font_size'     => '',
                'font_style'    => 'normal',
                'font_color'    => '',
            );

            $defaults = array(
                array_merge( $base, array(
                    'id'       => 'new_year',
                    'label'    => "New Year's Day",
                    'start'    => '01-01',
                    'end'      => '01-02',
                    'message'  => 'Happy New Year!',
                    'effect'   => 'snow',
                    'location' => 'top',
                    'bg_color' => '#0d1a33',
                    'enabled'  => 1,
                ) ),
                array_merge( $base, array(
                    'id'       => 'mlk_day',
                    'label'    => 'Martin Luther King Jr. Day',
                    'start'    => date( 'm-d', $mlk_ts ),
                    'end'      => date( 'm-d', $mlk_ts ),
                    'message'  => 'Honoring Martin Luther King Jr.',
                    'effect'   => 'flag',
                    'location' => 'top',
                    'bg_color' => '#192857',
                    'enabled'  => 1,
                ) ),
                array_merge( $base, array(
                    'id'       => 'presidents_day',
                    'label'    => 'Presidents Day',
                    'start'    => date( 'm-d', $presidents_ts ),
                    'end'      => date( 'm-d', $presidents_ts ),
                    'message'  => 'Happy Presidents Day.',
                    'effect'   => 'flag',
                    'location' => 'top',
                    'bg_color' => '#192857',
                    'enabled'  => 1,
                ) ),
                array_merge( $base, array(
                    'id'       => 'memorial_day',
                    'label'    => 'Memorial Day',
                    'start'    => date( 'm-d', strtotime( '-1 day', $memorial_ts ) ),
                    'end'      => date( 'm-d', strtotime( '+1 day', $memorial_ts ) ),
                    'message'  => 'Remembering and honoring those who gave all.',
                    'effect'   => 'flag',
                    'location' => 'top',
                    'bg_color' => '#192857',
                    'enabled'  => 1,
                ) ),
                array_merge( $base, array(
                    'id'       => 'independence_day',
                    'label'    => 'Independence Day',
                    'start'    => '07-03',
                    'end'      => '07-05',
                    'message'  => 'Happy 4th of July!',
                    'effect'   => 'flag',
                    'location' => 'top',
                    'bg_color' => '#b01f29',
                    'enabled'  => 1,
                ) ),
                array_merge( $base, array(
                    'id'       => 'labor_day',
                    'label'    => 'Labor Day',
                    'start'    => date( 'm-d', strtotime( '-1 day', $labor_ts ) ),
                    'end'      => date( 'm-d', strtotime( '+1 day', $labor_ts ) ),
                    'message'  => 'Happy Labor Day.',
                    'effect'   => 'leaves',
                    'location' => 'bottom',
                    'bg_color' => '#8b4513',
                    'enabled'  => 1,
                ) ),
                array_merge( $base, array(
                    'id'       => 'columbus_day',
                    'label'    => 'Columbus Day',
                    'start'    => '10-13',
                    'end'      => '10-13',
                    'message'  => 'Columbus Day.',
                    'effect'   => 'flag',
                    'location' => 'bottom',
                    'bg_color' => '#192857',
                    'enabled'  => 0,
                ) ),
                array_merge( $base, array(
                    'id'       => 'veterans_day',
                    'label'    => 'Veterans Day',
                    'start'    => '11-10',
                    'end'      => '11-12',
                    'message'  => 'Honoring all who served.',
                    'effect'   => 'flag',
                    'location' => 'top',
                    'bg_color' => '#192857',
                    'enabled'  => 1,
                ) ),
                array_merge( $base, array(
                    'id'       => 'thanksgiving_season',
                    'label'    => 'Thanksgiving Season',
                    'start'    => '11-01',
                    'end'      => date( 'm-d', strtotime( '-1 day', $thank_ts ) ),
                    'message'  => 'Getting ready for Thanksgiving!',
                    'effect'   => 'leaves',
                    'location' => 'top',
                    'bg_color' => '#8b4513',
                    'enabled'  => 1,
                ) ),
                array_merge( $base, array(
                    'id'       => 'thanksgiving_day',
                    'label'    => 'Thanksgiving Day',
                    'start'    => date( 'm-d', $thank_ts ),
                    'end'      => date( 'm-d', $thank_ts ),
                    'message'  => 'Happy Thanksgiving!',
                    'effect'   => 'leaves',
                    'location' => 'top',
                    'bg_color' => '#c65b1a',
                    'enabled'  => 1,
                ) ),
                array_merge( $base, array(
                    'id'       => 'christmas_season',
                    'label'    => 'Christmas Season',
                    'start'    => date( 'm-d', strtotime( '+1 day', $thank_ts ) ),
                    'end'      => '12-26',
                    'message'  => 'Getting ready for Christmas!',
                    'effect'   => 'snow',
                    'location' => 'top',
                    'bg_color' => '#192857',
                    'enabled'  => 1,
                ) ),
                array_merge( $base, array(
                    'id'       => 'christmas_day',
                    'label'    => 'Christmas Day',
                    'start'    => '12-25',
                    'end'      => '12-25',
                    'message'  => 'Merry Christmas!',
                    'effect'   => 'snow',
                    'location' => 'modal',
                    'bg_color' => '#192857',
                    'enabled'  => 1,
                ) ),
            );

            return $defaults;
        }

        private function get_default_settings() {
            return array(
                'enabled'       => 1,
                'banner_style'  => 'auto', 
                'global_note'   => '',
                'debug_enabled' => 0,
                'debug_date'    => '',
                'holidays'      => $this->get_default_holidays(),
				'assets_custom' => array(),

            );
        }

        private function get_settings() {
            $defaults = $this->get_default_settings();
            $saved    = get_option( $this->option_name, array() );
            if ( ! is_array( $saved ) ) {
                $saved = array();
            }

            $settings = array_merge( $defaults, $saved );

            if ( empty( $settings['holidays'] ) || ! is_array( $settings['holidays'] ) ) {
                $settings['holidays'] = $this->get_default_holidays();
            }

            return $settings;
        }

        public function register_settings() {
            register_setting(
                'ahs_settings_group',
                $this->option_name,
                array( $this, 'sanitize_settings' )
            );
        }

		private function build_groups_from_effect( $effect ) {
			$effect = sanitize_text_field( $effect );

			if ( $effect === 'none' ) {
				return array();
			}

			$allowed = array( 'snow', 'leaves', 'flag' );
			if ( ! in_array( $effect, $allowed, true ) ) {
				$effect = 'snow';
			}

			$count = ( $effect === 'snow' ) ? 120 : ( ( $effect === 'leaves' ) ? 80 : 60 );

			return array(
				array(
					'type'  => $effect,
					'count' => $count,
					'items' => array(),
				),
			);
		}

private function sanitize_effect_groups_json( $json, $fallback_effect = 'snow' ) {
    $json = trim( (string) $json );

    if ( $json === '' ) {
        return $this->build_groups_from_effect( $fallback_effect );
    }

    $data = json_decode( $json, true );
    if ( ! is_array( $data ) ) {
        return $this->build_groups_from_effect( $fallback_effect );
    }

    $clean = array();
    $allowed_types = array( 'snow', 'leaves', 'flag', 'custom' );

    foreach ( $data as $g ) {
        if ( ! is_array( $g ) ) {
            continue;
        }

        $type = isset( $g['type'] ) ? sanitize_text_field( $g['type'] ) : 'snow';
        if ( ! in_array( $type, $allowed_types, true ) ) {
            $type = 'snow';
        }

        $count = isset( $g['count'] ) ? intval( $g['count'] ) : 0;
        if ( $count <= 0 ) {
            $count = ( $type === 'snow' ) ? 120 : ( ( $type === 'leaves' ) ? 80 : 60 );
        }
        if ( $count > 400 ) {
            $count = 400;
        }

        $items = array();

if ( isset( $g['items'] ) ) {
    $raw_items = $g['items'];

    if ( is_string( $raw_items ) ) {
        $raw_items = trim( $raw_items );
        $raw_items = ( $raw_items !== '' ) ? array( $raw_items ) : array();
    } elseif ( ! is_array( $raw_items ) ) {
        $raw_items = array();
    }

    foreach ( $raw_items as $u ) {
        $u = esc_url_raw( $u );
        if ( $u !== '' ) {
            $items[] = $u;
        }
    }
}
        $clean[] = array(
            'type'  => $type,
            'count' => $count,
            'items' => $items,
        );
    }

    if ( empty( $clean ) ) {
        return $this->build_groups_from_effect( $fallback_effect );
    }

    return $clean;
}

private function resolve_effect_groups_for_today( $settings, $holiday ) {
    
    if ( isset( $settings['banner_style'] ) && $settings['banner_style'] !== 'auto' ) {
        return $this->build_groups_from_effect( $settings['banner_style'] );
    }

    
    if ( isset( $holiday['effect_groups'] ) && is_array( $holiday['effect_groups'] ) && ! empty( $holiday['effect_groups'] ) ) {
        return $holiday['effect_groups'];
    }

   
    $legacy = isset( $holiday['effect'] ) ? $holiday['effect'] : 'snow';
    return $this->build_groups_from_effect( $legacy );
}

        private function normalize_md( $value ) {
            $value = trim( $value );
            $value = str_replace( '/', '-', $value );

            if ( preg_match( '/^(\d{1,2})-(\d{1,2})$/', $value, $m ) ) {
                $m1 = (int) $m[1];
                $d1 = (int) $m[2];
                if ( $m1 < 1 || $m1 > 12 || $d1 < 1 || $d1 > 31 ) {
                    return '';
                }
                return sprintf( '%02d-%02d', $m1, $d1 );
            }

            return '';
        }

			   private function sanitize_background_fields( $row ) {
			$bg_color = isset( $row['bg_color'] ) ? sanitize_text_field( $row['bg_color'] ) : '';
			$bg_color = trim( $bg_color );

			if ( '' !== $bg_color && strtolower( $bg_color ) === 'transparent' ) {
				$bg_color = '';
			}

			$bg_image = isset( $row['bg_image'] ) ? esc_url_raw( $row['bg_image'] ) : '';

			$bg_repeat     = isset( $row['bg_repeat'] ) ? sanitize_text_field( $row['bg_repeat'] ) : '';
			$bg_position   = isset( $row['bg_position'] ) ? sanitize_text_field( $row['bg_position'] ) : '';
			$bg_size       = isset( $row['bg_size'] ) ? sanitize_text_field( $row['bg_size'] ) : '';
			$banner_height = isset( $row['banner_height'] ) ? sanitize_text_field( $row['banner_height'] ) : '';
			$bg_opacity    = isset( $row['bg_opacity'] ) ? $row['bg_opacity'] : '';
			$overlap       = isset( $row['overlap'] ) ? $row['overlap'] : '';
			$bg_offset_x   = isset( $row['bg_offset_x'] ) ? $row['bg_offset_x'] : '';
    		$bg_offset_y   = isset( $row['bg_offset_y'] ) ? $row['bg_offset_y'] : '';

			$allowed_repeat   = array( 'no-repeat', 'repeat', 'repeat-x', 'repeat-y' );
			$allowed_position = array(
				'left top', 'center top', 'right top',
				'left bottom', 'center bottom', 'right bottom',
				'center center',
			);
			$allowed_size = array( 'auto', 'cover', 'contain' );

			if ( ! in_array( $bg_repeat, $allowed_repeat, true ) ) {
				$bg_repeat = 'no-repeat';
			}
			if ( ! in_array( $bg_position, $allowed_position, true ) ) {
				$bg_position = 'center top';
			}
			if ( ! in_array( $bg_size, $allowed_size, true ) ) {
				$bg_size = 'cover';
			}

			$banner_height = preg_replace( '/[^0-9]/', '', $banner_height );

			$bg_opacity = preg_replace( '/[^0-9]/', '', $bg_opacity );
			if ( $bg_opacity === '' ) {
				$bg_opacity = '';
			} else {
				$bg_opacity = (int) $bg_opacity;
				if ( $bg_opacity < 0 || $bg_opacity > 100 ) {
					$bg_opacity = '';
				}
			}

			$overlap = preg_replace( '/[^0-9\-]/', '', $overlap );
				   
			$bg_offset_x = preg_replace( '/[^0-9\-]/', '', $bg_offset_x );
   			$bg_offset_y = preg_replace( '/[^0-9\-]/', '', $bg_offset_y );
				   
			return array(
				'bg_color'      => $bg_color,
				'bg_image'      => $bg_image,
				'bg_repeat'     => $bg_repeat,
				'bg_position'   => $bg_position,
				'bg_size'       => $bg_size,
				'banner_height' => $banner_height,
				'bg_opacity'    => $bg_opacity,
				'overlap'       => $overlap,
				'bg_offset_x'   => $bg_offset_x,
        		'bg_offset_y'   => $bg_offset_y,
			);
		}

        public function sanitize_settings( $input ) {
            $defaults = $this->get_default_settings();
            $output   = $this->settings;

            $output['enabled'] = ! empty( $input['enabled'] ) ? 1 : 0;

            $allowed_styles = array( 'auto', 'snow', 'leaves', 'flag', 'none' );
            if ( isset( $input['banner_style'] ) && in_array( $input['banner_style'], $allowed_styles, true ) ) {
                $output['banner_style'] = $input['banner_style'];
            } else {
                $output['banner_style'] = $defaults['banner_style'];
            }

            if ( isset( $input['global_note'] ) ) {
                $output['global_note'] = wp_kses_post( $input['global_note'] );
            } else {
                $output['global_note'] = '';
            }

            $output['debug_enabled'] = ! empty( $input['debug_enabled'] ) ? 1 : 0;

            if ( isset( $input['debug_date'] ) && ! empty( $input['debug_date'] ) ) {
                $date = sanitize_text_field( $input['debug_date'] );
                if ( preg_match( '/^\d{4}-\d{2}-\d{2}$/', $date ) ) {
                    $output['debug_date'] = $date;
                } else {
                    $output['debug_date'] = '';
                }
            } else {
                $output['debug_date'] = '';
            }
			
			$output['assets_custom'] = array();
			if ( isset( $input['assets_custom'] ) && is_array( $input['assets_custom'] ) ) {
				foreach ( $input['assets_custom'] as $a ) {
					if ( ! is_array( $a ) ) { continue; }
					$name = isset( $a['name'] ) ? sanitize_text_field( $a['name'] ) : '';
					$url  = isset( $a['url'] ) ? esc_url_raw( $a['url'] ) : '';
					if ( $name === '' && $url === '' ) { continue; }
					$output['assets_custom'][] = array(
						'name' => $name,
						'url'  => $url,
					);
				}
			}

            $holidays_clean = array();

            if ( isset( $input['holidays'] ) && is_array( $input['holidays'] ) ) {
                foreach ( $input['holidays'] as $row ) {
                    $delete  = ! empty( $row['delete'] );
                    $enabled = ! empty( $row['enabled'] ) ? 1 : 0;

                    $id       = isset( $row['id'] ) ? sanitize_key( $row['id'] ) : '';
                    $label    = isset( $row['label'] ) ? sanitize_text_field( $row['label'] ) : '';
                    $start    = isset( $row['start'] ) ? sanitize_text_field( $row['start'] ) : '';
                    $end      = isset( $row['end'] ) ? sanitize_text_field( $row['end'] ) : '';
                    $message  = isset( $row['message'] ) ? wp_kses_post( $row['message'] ) : '';
                    $effect   = isset( $row['effect'] ) ? sanitize_text_field( $row['effect'] ) : '';
					$groups_json  = isset( $row['effect_groups_json'] ) ? wp_unslash( $row['effect_groups_json'] ) : '';
					$effect_groups = $this->sanitize_effect_groups_json( $groups_json, $effect );

                    $location = isset( $row['location'] ) ? sanitize_text_field( $row['location'] ) : '';

                    if ( $delete ) {
                        continue;
                    }

                    if ( '' === $label || '' === $start || '' === $end ) {
                        continue;
                    }

                    $start = $this->normalize_md( $start );
                    $end   = $this->normalize_md( $end );

                    if ( ! $start || ! $end ) {
                        continue;
                    }

                    if ( '' === $id ) {
                        $id = sanitize_key( $label );
                    }

                    $allowed_effects   = array( 'snow', 'leaves', 'flag', 'none' );
                    $allowed_locations = array( 'top', 'bottom', 'modal' );

                    if ( ! in_array( $effect, $allowed_effects, true ) ) {
                        $effect = 'snow';
                    }
                    if ( ! in_array( $location, $allowed_locations, true ) ) {
                        $location = 'top';
                    }

                    $bg = $this->sanitize_background_fields( $row );
					
					$text_h_align  = isset( $row['text_h_align'] ) ? sanitize_text_field( $row['text_h_align'] ) : 'center';
					$text_v_align  = isset( $row['text_v_align'] ) ? sanitize_text_field( $row['text_v_align'] ) : 'middle';
					$text_offset_x = isset( $row['text_offset_x'] ) ? $row['text_offset_x'] : '';
					$text_offset_y = isset( $row['text_offset_y'] ) ? $row['text_offset_y'] : '';

					$allowed_h = array( 'left', 'center', 'right' );
					if ( ! in_array( $text_h_align, $allowed_h, true ) ) {
						$text_h_align = 'center';
					}

					$allowed_v = array( 'top', 'middle', 'bottom' );
					if ( ! in_array( $text_v_align, $allowed_v, true ) ) {
						$text_v_align = 'middle';
					}

					$text_offset_x = preg_replace( '/[^0-9\-]/', '', $text_offset_x );
					$text_offset_y = preg_replace( '/[^0-9\-]/', '', $text_offset_y );
					
					$font_family = isset( $row['font_family'] ) ? sanitize_text_field( $row['font_family'] ) : '';
					$font_size   = isset( $row['font_size'] ) ? $row['font_size'] : '';
					$font_style  = isset( $row['font_style'] ) ? sanitize_text_field( $row['font_style'] ) : 'normal';
					$font_color  = isset( $row['font_color'] ) ? sanitize_hex_color( $row['font_color'] ) : '';

					$font_size = preg_replace( '/[^0-9]/', '', $font_size );

					$allowed_font_styles = array( 'normal', 'bold', 'italic', 'bold_italic' );
					if ( ! in_array( $font_style, $allowed_font_styles, true ) ) {
						$font_style = 'normal';
					}

					
                    $holidays_clean[] = array(
                        'id'            => $id,
                        'label'         => $label,
                        'start'         => $start,
                        'end'           => $end,
                        'message'       => $message,
                        'effect'        => $effect,
                        'location'      => $location,
                        'bg_color'      => $bg['bg_color'],
                        'bg_image'      => $bg['bg_image'],
                        'bg_repeat'     => $bg['bg_repeat'],
                        'bg_position'   => $bg['bg_position'],
                        'bg_size'       => $bg['bg_size'],
                        'banner_height' => $bg['banner_height'],
						'bg_opacity'    => $bg['bg_opacity'],
    					'overlap'       => $bg['overlap'],
						'bg_offset_x'   => $bg['bg_offset_x'],
						'bg_offset_y'   => $bg['bg_offset_y'],
                        'enabled'       => $enabled,
						'text_h_align'  => $text_h_align,
						'text_v_align'  => $text_v_align,
						'text_offset_x' => $text_offset_x,
						'text_offset_y' => $text_offset_y,
						'font_family'   => $font_family,
                        'font_size'     => $font_size,
                        'font_style'    => $font_style,
                        'font_color'    => $font_color,
						'effect_groups' => $effect_groups,
                    );
                }
            }

            if ( isset( $input['holidays_new'] ) && is_array( $input['holidays_new'] ) ) {
                $row      = $input['holidays_new'];
                $enabled  = ! empty( $row['enabled'] ) ? 1 : 0;
                $id       = isset( $row['id'] ) ? sanitize_key( $row['id'] ) : '';
                $label    = isset( $row['label'] ) ? sanitize_text_field( $row['label'] ) : '';
                $start    = isset( $row['start'] ) ? sanitize_text_field( $row['start'] ) : '';
                $end      = isset( $row['end'] ) ? sanitize_text_field( $row['end'] ) : '';
                $message  = isset( $row['message'] ) ? wp_kses_post( $row['message'] ) : '';
                $effect   = isset( $row['effect'] ) ? sanitize_text_field( $row['effect'] ) : '';
                $location = isset( $row['location'] ) ? sanitize_text_field( $row['location'] ) : '';

                if ( '' !== $label && '' !== $start && '' !== $end ) {
                    $start = $this->normalize_md( $start );
                    $end   = $this->normalize_md( $end );

                    if ( $start && $end ) {
                        if ( '' === $id ) {
                            $id = sanitize_key( $label );
                        }

                        $allowed_effects   = array( 'snow', 'leaves', 'flag', 'none' );
                        $allowed_locations = array( 'top', 'bottom', 'modal' );

                        if ( ! in_array( $effect, $allowed_effects, true ) ) {
                            $effect = 'snow';
                        }
                        if ( ! in_array( $location, $allowed_locations, true ) ) {
                            $location = 'top';
                        }

                        $bg = $this->sanitize_background_fields( $row );
						
						$text_h_align  = isset( $row['text_h_align'] ) ? sanitize_text_field( $row['text_h_align'] ) : 'center';
						$text_v_align  = isset( $row['text_v_align'] ) ? sanitize_text_field( $row['text_v_align'] ) : 'middle';
						$text_offset_x = isset( $row['text_offset_x'] ) ? $row['text_offset_x'] : '';
						$text_offset_y = isset( $row['text_offset_y'] ) ? $row['text_offset_y'] : '';

						$allowed_h = array( 'left', 'center', 'right' );
						if ( ! in_array( $text_h_align, $allowed_h, true ) ) {
							$text_h_align = 'center';
						}

						$allowed_v = array( 'top', 'middle', 'bottom' );
						if ( ! in_array( $text_v_align, $allowed_v, true ) ) {
							$text_v_align = 'middle';
						}

						$text_offset_x = preg_replace( '/[^0-9\-]/', '', $text_offset_x );
						$text_offset_y = preg_replace( '/[^0-9\-]/', '', $text_offset_y );

						$font_family = isset( $row['font_family'] ) ? sanitize_text_field( $row['font_family'] ) : '';
						$font_size   = isset( $row['font_size'] ) ? $row['font_size'] : '';
						$font_style  = isset( $row['font_style'] ) ? sanitize_text_field( $row['font_style'] ) : 'normal';
						$font_color  = isset( $row['font_color'] ) ? sanitize_hex_color( $row['font_color'] ) : '';

						$font_size = preg_replace( '/[^0-9]/', '', $font_size );

						$allowed_font_styles = array( 'normal', 'bold', 'italic', 'bold_italic' );
						if ( ! in_array( $font_style, $allowed_font_styles, true ) ) {
							$font_style = 'normal';
						}
						
                        $holidays_clean[] = array(
                            'id'            => $id,
                            'label'         => $label,
                            'start'         => $start,
                            'end'           => $end,
                            'message'       => $message,
                            'effect'        => $effect,
                            'location'      => $location,
                            'bg_color'      => $bg['bg_color'],
                            'bg_image'      => $bg['bg_image'],
                            'bg_repeat'     => $bg['bg_repeat'],
                            'bg_position'   => $bg['bg_position'],
                            'bg_size'       => $bg['bg_size'],
                            'banner_height' => $bg['banner_height'],
							'bg_opacity'    => $bg['bg_opacity'],
    						'overlap'       => $bg['overlap'],
                            'enabled'       => $enabled,
							'text_h_align'  => $text_h_align,
							'text_v_align'  => $text_v_align,
							'text_offset_x' => $text_offset_x,
							'text_offset_y' => $text_offset_y,
							'font_family'   => $font_family,
							'font_size'     => $font_size,
							'font_style'    => $font_style,
							'font_color'    => $font_color,
                        );
                    }
                }
            }

            if ( isset( $input['holidays_json_import'] ) && trim( $input['holidays_json_import'] ) !== '' ) {
                $raw     = wp_unslash( $input['holidays_json_import'] );
                $raw     = trim( $raw );
                $decoded = json_decode( $raw, true );

                if ( json_last_error() === JSON_ERROR_NONE && is_array( $decoded ) ) {
                    $holidays_clean = array();
                    foreach ( $decoded as $row ) {
                        if ( ! is_array( $row ) ) {
                            continue;
                        }

                        $enabled  = ! empty( $row['enabled'] ) ? 1 : 0;
                        $id       = isset( $row['id'] ) ? sanitize_key( $row['id'] ) : '';
                        $label    = isset( $row['label'] ) ? sanitize_text_field( $row['label'] ) : '';
                        $start    = isset( $row['start'] ) ? sanitize_text_field( $row['start'] ) : '';
                        $end      = isset( $row['end'] ) ? sanitize_text_field( $row['end'] ) : '';
                        $message  = isset( $row['message'] ) ? wp_kses_post( $row['message'] ) : '';
                        $effect   = isset( $row['effect'] ) ? sanitize_text_field( $row['effect'] ) : '';
                        $location = isset( $row['location'] ) ? sanitize_text_field( $row['location'] ) : '';

                        if ( '' === $label || '' === $start || '' === $end ) {
                            continue;
                        }

                        $start = $this->normalize_md( $start );
                        $end   = $this->normalize_md( $end );
                        if ( ! $start || ! $end ) {
                            continue;
                        }

                        if ( '' === $id ) {
                            $id = sanitize_key( $label );
                        }

                        $allowed_effects   = array( 'snow', 'leaves', 'flag', 'none' );
                        $allowed_locations = array( 'top', 'bottom', 'modal' );

                        if ( ! in_array( $effect, $allowed_effects, true ) ) {
                            $effect = 'snow';
                        }
                        if ( ! in_array( $location, $allowed_locations, true ) ) {
                            $location = 'top';
                        }

                        $bg = $this->sanitize_background_fields( $row );
							
						$text_h_align  = isset( $row['text_h_align'] ) ? sanitize_text_field( $row['text_h_align'] ) : 'center';
						$text_v_align  = isset( $row['text_v_align'] ) ? sanitize_text_field( $row['text_v_align'] ) : 'middle';
						$text_offset_x = isset( $row['text_offset_x'] ) ? $row['text_offset_x'] : '';
						$text_offset_y = isset( $row['text_offset_y'] ) ? $row['text_offset_y'] : '';

						$allowed_h = array( 'left', 'center', 'right' );
						if ( ! in_array( $text_h_align, $allowed_h, true ) ) {
							$text_h_align = 'center';
						}

						$allowed_v = array( 'top', 'middle', 'bottom' );
						if ( ! in_array( $text_v_align, $allowed_v, true ) ) {
							$text_v_align = 'middle';
						}

						$text_offset_x = preg_replace( '/[^0-9\-]/', '', $text_offset_x );
						$text_offset_y = preg_replace( '/[^0-9\-]/', '', $text_offset_y );

						$font_family = isset( $row['font_family'] ) ? sanitize_text_field( $row['font_family'] ) : '';
						$font_size   = isset( $row['font_size'] ) ? $row['font_size'] : '';
						$font_style  = isset( $row['font_style'] ) ? sanitize_text_field( $row['font_style'] ) : 'normal';
						$font_color  = isset( $row['font_color'] ) ? sanitize_hex_color( $row['font_color'] ) : '';

						$font_size = preg_replace( '/[^0-9]/', '', $font_size );

						$allowed_font_styles = array( 'normal', 'bold', 'italic', 'bold_italic' );
						if ( ! in_array( $font_style, $allowed_font_styles, true ) ) {
							$font_style = 'normal';
						}

                        $holidays_clean[] = array(
                            'id'            => $id,
                            'label'         => $label,
                            'start'         => $start,
                            'end'           => $end,
                            'message'       => $message,
                            'effect'        => $effect,
                            'location'      => $location,
                            'bg_color'      => $bg['bg_color'],
                            'bg_image'      => $bg['bg_image'],
                            'bg_repeat'     => $bg['bg_repeat'],
                            'bg_position'   => $bg['bg_position'],
                            'bg_size'       => $bg['bg_size'],
                            'banner_height' => $bg['banner_height'],
							'bg_opacity'    => $bg['bg_opacity'],
    						'overlap'       => $bg['overlap'],
                            'enabled'       => $enabled,
							'text_h_align'  => $text_h_align,
							'text_v_align'  => $text_v_align,
							'text_offset_x' => $text_offset_x,
							'text_offset_y' => $text_offset_y,
							'font_family'   => $font_family,
							'font_size'     => $font_size,
							'font_style'    => $font_style,
							'font_color'    => $font_color,
                        );
                    }
                }
            }

            if ( empty( $holidays_clean ) ) {
                $holidays_clean = $this->get_default_holidays();
            }

            $output['holidays'] = $holidays_clean;

            if ( isset( $output['holidays_json_import'] ) ) {
                unset( $output['holidays_json_import'] );
            }

            return $output;
        }

        public function add_settings_page() {

			add_menu_page(
				'Celeb-Right',
				'Celeb-Right',
				'manage_options',
				'ahs-holiday-effects',
				array( $this, 'render_settings_page' ),
				'dashicons-star-filled',
				58
			);

			add_submenu_page(
				'ahs-holiday-effects',
				'Settings',
				'Settings',
				'manage_options',
				'ahs-holiday-effects',
				array( $this, 'render_settings_page' )
			);
		}

        public function render_settings_page() {
            if ( ! current_user_can( 'manage_options' ) ) {
                return;
            }

            $this->settings = $this->get_settings();
            $holidays       = $this->settings['holidays'];
            $export_json    = wp_json_encode( $holidays, JSON_PRETTY_PRINT );
            ?>
           <div class="wrap celeb-right-admin">
                <h1>Celeb-Right Banner & Effects</h1>
                <form method="post" action="options.php">
                    <?php settings_fields( 'ahs_settings_group' ); ?>

                    <h2 class="title">General settings</h2>

			<div class="cr-panel">
				<div class="cr-grid">
					<!-- Enable -->
					<div class="cr-field">
						<div class="cr-field-top">
							<label class="cr-label" for="ahs_enabled">Enable holiday banner</label>

							<label class="cr-switch" title="Enable holiday banner">
								<input type="checkbox"
									   name="<?php echo esc_attr( $this->option_name ); ?>[enabled]"
									   id="ahs_enabled"
									   value="1"
									<?php checked( $this->settings['enabled'], 1 ); ?> />
								<span class="cr-slider" aria-hidden="true"></span>
							</label>
						</div>
						<div class="cr-help">Show banner/effects automatically when a holiday is active.</div>
					</div>

					<!-- Global override -->
					<div class="cr-field">
						<label class="cr-label" for="ahs_banner_style">Global effect override</label>
						<select name="<?php echo esc_attr( $this->option_name ); ?>[banner_style]" id="ahs_banner_style">
							<option value="auto"  <?php selected( $this->settings['banner_style'], 'auto' ); ?>>Use each holiday's effect</option>
							<option value="snow"  <?php selected( $this->settings['banner_style'], 'snow' ); ?>>Force snow for all</option>
							<option value="leaves"<?php selected( $this->settings['banner_style'], 'leaves' ); ?>>Force leaves for all</option>
							<option value="flag"  <?php selected( $this->settings['banner_style'], 'flag' ); ?>>Force flags for all</option>
							<option value="none"  <?php selected( $this->settings['banner_style'], 'none' ); ?>>No falling objects (banner/modal only)</option>
						</select>
						<?php
							$export_json = wp_json_encode( $this->settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES );
						?>

						<div class="cr-actions cr-actions--compact">
							<button type="button" class="button" id="cr_export_json" data-export="<?php echo esc_attr( $export_json ); ?>">Export</button>

							<input type="file" id="cr_import_json_file" accept="application/json" style="display:none;" />
							<button type="button" class="button" id="cr_import_json_choose">Import</button>

							<button type="button" class="button" id="cr_import_json_clear">Clear</button>

							<span class="cr-status" id="cr_import_status"></span>
						</div>

						<textarea name="<?php echo esc_attr( $this->option_name ); ?>[holidays_json_import]"
								  id="ahs_holidays_json_import"
								  style="display:none;">
						</textarea>
					</div>
					

					<!-- Global note -->
					<div class="cr-field">
						<label class="cr-label" for="ahs_global_note">Global note</label>
						<textarea
							name="<?php echo esc_attr( $this->option_name ); ?>[global_note]"
							id="ahs_global_note"
							rows="3"
							class="cr-textarea-fixed"
						><?php echo esc_textarea( $this->settings['global_note'] ); ?></textarea>
						<div class="cr-help">Shown under the holiday banner/message on every page.</div>
					</div>


					<!-- Debug mode -->
					<div class="cr-field" id="cr_debug_wrap">
						<div class="cr-field-top">
							<label class="cr-label" for="ahs_debug_enabled">Debug mode</label>
							<label class="cr-switch">
								<input type="checkbox"
									   name="<?php echo esc_attr( $this->option_name ); ?>[debug_enabled]"
									   id="ahs_debug_enabled"
									   value="1"
									   <?php checked( $this->settings['debug_enabled'], 1 ); ?> />
								<span class="cr-slider"></span>
							</label>
						</div>
						<div class="cr-help">Use a custom date instead of today's date on the front-end.</div>

						<div class="cr-debug-date" id="cr_debug_date_wrap">
							<label class="cr-label" for="ahs_debug_date">Debug date</label>
							<input type="text"
								   name="<?php echo esc_attr( $this->option_name ); ?>[debug_date]"
								   id="ahs_debug_date"
								   value="<?php echo esc_attr( $this->settings['debug_date'] ); ?>"
								   placeholder="2025-12-25" />
							<div class="cr-help">Format: <code>YYYY-MM-DD</code>. Converted to MM-DD and matched against the holiday ranges.</div>
						</div>
					</div>


				</div>
			</div>

			<h2 class="title">Asset Library</h2>
			<p class="description">
				Add extra SVG/PNG assets (gummy bears, other flags, etc.). You can reuse these inside any holiday via “Falling groups”.
			</p>

			<?php
			$assets_custom = isset( $this->settings['assets_custom'] ) && is_array( $this->settings['assets_custom'] )
				? $this->settings['assets_custom']
				: array();
			?>

			<table class="widefat striped ahs-assets-table" id="ahs_assets_table">
				<thead>
					<tr>
						<th style="width:220px;">Name</th>
						<th>Asset URL</th>
						<th style="width:90px;">Upload</th>
						<th style="width:80px;">Remove</th>
					</tr>
				</thead>
				<tbody>
				<?php if ( ! empty( $assets_custom ) ) : ?>
					<?php foreach ( $assets_custom as $a_i => $a ) :
						$a_name = isset( $a['name'] ) ? $a['name'] : '';
						$a_url  = isset( $a['url'] ) ? $a['url'] : '';
						?>
						<tr class="ahs-asset-row" data-idx="<?php echo esc_attr( $a_i ); ?>">
							<td>
								<input type="text"
									class="regular-text ahs-asset-name"
									name="<?php echo esc_attr( $this->option_name ); ?>[assets_custom][<?php echo esc_attr( $a_i ); ?>][name]"
									value="<?php echo esc_attr( $a_name ); ?>" />
							</td>
							<td>
								<input type="text"
									class="regular-text ahs-asset-url"
									style="width:100%;"
									name="<?php echo esc_attr( $this->option_name ); ?>[assets_custom][<?php echo esc_attr( $a_i ); ?>][url]"
									value="<?php echo esc_attr( $a_url ); ?>"
									placeholder="https://example.com/asset.svg" />
							</td>
							<td>
								<button type="button" class="button ahs-asset-upload">Upload</button>
							</td>
							<td>
								<button type="button" class="button-link-delete ahs-asset-remove">Remove</button>
							</td>
						</tr>
					<?php endforeach; ?>
				<?php else : ?>
					<tr class="ahs-asset-row" data-idx="0">
						<td>
							<input type="text"
								class="regular-text ahs-asset-name"
								name="<?php echo esc_attr( $this->option_name ); ?>[assets_custom][0][name]"
								value=""
								placeholder="Gummy Bear (Red)" />
						</td>
						<td>
							<input type="text"
								class="regular-text ahs-asset-url"
								style="width:100%;"
								name="<?php echo esc_attr( $this->option_name ); ?>[assets_custom][0][url]"
								value=""
								placeholder="https://example.com/gummy-bear.svg" />
						</td>
						<td>
							<button type="button" class="button ahs-asset-upload">Upload</button>
						</td>
						<td>
							<button type="button" class="button-link-delete ahs-asset-remove">Remove</button>
						</td>
					</tr>
				<?php endif; ?>
				</tbody>
			</table>

			<p>
				<button type="button" class="button button-secondary" id="ahs_add_asset">Add asset</button>
			</p>

					
                   <h2 class="title">Holiday list</h2>
				<p>Each row is one holiday/season. Dates use <strong>MM-DD</strong> (for example <code>11-01</code>, <code>12-25</code>).</p>

				<table class="widefat striped ahs-holiday-table">
					<thead>
						<tr>
							<th style="width:40px;">On</th>
							<th>Label</th>
							<th style="width:90px;">Start</th>
							<th style="width:90px;">End</th>
							<th style="width:80px;">Edit</th>
							<th style="width:60px;">Delete</th>
						</tr>
					</thead>
					<tbody>
					<?php foreach ( $holidays as $index => $holiday ) : ?>
						<?php
						$bg_color      = isset( $holiday['bg_color'] ) ? $holiday['bg_color'] : '';
						$bg_image      = isset( $holiday['bg_image'] ) ? $holiday['bg_image'] : '';
						$bg_repeat     = isset( $holiday['bg_repeat'] ) ? $holiday['bg_repeat'] : 'no-repeat';
						$bg_position   = isset( $holiday['bg_position'] ) ? $holiday['bg_position'] : 'center top';
						$bg_size       = isset( $holiday['bg_size'] ) ? $holiday['bg_size'] : 'cover';
						$banner_height = isset( $holiday['banner_height'] ) ? $holiday['banner_height'] : '';
						$overlap       = isset( $holiday['overlap'] ) ? $holiday['overlap'] : '';
						$bg_opacity    = isset( $holiday['bg_opacity'] ) ? $holiday['bg_opacity'] : '';
						$bg_offset_x   = isset( $holiday['bg_offset_x'] ) ? $holiday['bg_offset_x'] : '';
						$bg_offset_y   = isset( $holiday['bg_offset_y'] ) ? $holiday['bg_offset_y'] : '';
						$text_h_align  = isset( $holiday['text_h_align'] ) ? $holiday['text_h_align'] : 'center';
						$text_v_align  = isset( $holiday['text_v_align'] ) ? $holiday['text_v_align'] : 'middle';
						$text_offset_x = isset( $holiday['text_offset_x'] ) ? $holiday['text_offset_x'] : '';
						$text_offset_y = isset( $holiday['text_offset_y'] ) ? $holiday['text_offset_y'] : '';
						$font_family   = isset( $holiday['font_family'] ) ? $holiday['font_family'] : '';
						$font_size     = isset( $holiday['font_size'] ) ? $holiday['font_size'] : '';
						$font_style    = isset( $holiday['font_style'] ) ? $holiday['font_style'] : 'normal';
						$font_color    = isset( $holiday['font_color'] ) ? $holiday['font_color'] : '';

						?>
						<tr class="ahs-holiday-row" data-row="<?php echo esc_attr( $index ); ?>">
							<td>
								<input type="hidden"
							   class="ahs-field-text-h"
							   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][text_h_align]"
							   value="<?php echo esc_attr( $text_h_align ); ?>" />

						<input type="hidden"
							   class="ahs-field-text-v"
							   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][text_v_align]"
							   value="<?php echo esc_attr( $text_v_align ); ?>" />

						<input type="hidden"
							   class="ahs-field-text-offset-x"
							   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][text_offset_x]"
							   value="<?php echo esc_attr( $text_offset_x ); ?>" />

						<input type="hidden"
							   class="ahs-field-text-offset-y"
							   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][text_offset_y]"
							   value="<?php echo esc_attr( $text_offset_y ); ?>" />
								<input type="checkbox"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][enabled]"
									   value="1"
									<?php checked( ! empty( $holiday['enabled'] ), 1 ); ?> />
							</td>
							<td>
								<input type="text"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][label]"
									   value="<?php echo esc_attr( $holiday['label'] ); ?>"
									   style="width:100%;" />
								<input type="hidden"
									   class="ahs-field-id"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][id]"
									   value="<?php echo esc_attr( $holiday['id'] ); ?>" />
							</td>
							<td>
								<input type="text"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][start]"
									   value="<?php echo esc_attr( $holiday['start'] ); ?>"
									   style="width:100%;" />
							</td>
							<td>
								<input type="text"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][end]"
									   value="<?php echo esc_attr( $holiday['end'] ); ?>"
									   style="width:100%;" />
							</td>
							<td>
								<button type="button"
										class="button button-secondary ahs-edit-btn"
										data-row="<?php echo esc_attr( $index ); ?>">
									Edit
								</button>
							</td>
							<td style="text-align:center;">
								<input type="checkbox"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][delete]"
									   value="1" />
							</td>
						</tr>

						<!-- Hidden advanced fields for this row (edited via modal) -->
						<tr class="ahs-hidden-row" data-row="<?php echo esc_attr( $index ); ?>" style="display:none;">
							<td colspan="6">
								<input type="hidden"
									   class="ahs-field-message"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][message]"
									   value="<?php echo esc_attr( $holiday['message'] ); ?>" />

								<input type="hidden"
									   class="ahs-field-effect"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][effect]"
									   value="<?php echo esc_attr( $holiday['effect'] ); ?>" />
								
									<?php
									$groups_val = array();
									if ( isset( $holiday['effect_groups'] ) && is_array( $holiday['effect_groups'] ) ) {
										$groups_val = $holiday['effect_groups'];
									} else {
										$groups_val = $this->build_groups_from_effect( $holiday['effect'] );
									}
									?>
									<input type="hidden"
										   class="ahs-field-effect-groups"
										   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][effect_groups_json]"
										   value="<?php echo esc_attr( wp_json_encode( $groups_val ) ); ?>" />


								<input type="hidden"
									   class="ahs-field-location"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][location]"
									   value="<?php echo esc_attr( $holiday['location'] ); ?>" />

								<input type="hidden"
									   class="ahs-field-bg-color"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][bg_color]"
									   value="<?php echo esc_attr( $bg_color ); ?>" />

								<input type="hidden"
									   class="ahs-field-bg-image"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][bg_image]"
									   value="<?php echo esc_attr( $bg_image ); ?>" />

								<input type="hidden"
									   class="ahs-field-bg-repeat"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][bg_repeat]"
									   value="<?php echo esc_attr( $bg_repeat ); ?>" />

								<input type="hidden"
									   class="ahs-field-bg-position"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][bg_position]"
									   value="<?php echo esc_attr( $bg_position ); ?>" />

								<input type="hidden"
									   class="ahs-field-bg-size"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][bg_size]"
									   value="<?php echo esc_attr( $bg_size ); ?>" />

								<input type="hidden"
									   class="ahs-field-banner-height"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][banner_height]"
									   value="<?php echo esc_attr( $banner_height ); ?>" />
								
								<input type="hidden"
									   class="ahs-field-overlap"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][overlap]"
									   value="<?php echo esc_attr( $overlap ); ?>" />

								<input type="hidden"
									   class="ahs-field-bg-opacity"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][bg_opacity]"
									   value="<?php echo esc_attr( $bg_opacity ); ?>" />
								
								<input type="hidden"
									   class="ahs-field-bg-offset-x"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][bg_offset_x]"
									   value="<?php echo esc_attr( $bg_offset_x ); ?>" />

								<input type="hidden"
									   class="ahs-field-bg-offset-y"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][bg_offset_y]"
									   value="<?php echo esc_attr( $bg_offset_y ); ?>" />

								<input type="hidden"
									   class="ahs-field-font-family"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][font_family]"
									   value="<?php echo esc_attr( $font_family ); ?>" />

								<input type="hidden"
									   class="ahs-field-font-size"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][font_size]"
									   value="<?php echo esc_attr( $font_size ); ?>" />

								<input type="hidden"
									   class="ahs-field-font-style"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][font_style]"
									   value="<?php echo esc_attr( $font_style ); ?>" />

								<input type="hidden"
									   class="ahs-field-font-color"
									   name="<?php echo esc_attr( $this->option_name ); ?>[holidays][<?php echo esc_attr( $index ); ?>][font_color]"
									   value="<?php echo esc_attr( $font_color ); ?>" />

							</td>
						</tr>
					<?php endforeach; ?>

					<!-- New holiday row (simple; advanced settings can be edited after save) -->
					<tr class="ahs-holiday-row-new">
						<td>
							<input type="checkbox"
								   name="<?php echo esc_attr( $this->option_name ); ?>[holidays_new][enabled]"
								   value="1" />
						</td>
						<td>
							<input type="text"
								   name="<?php echo esc_attr( $this->option_name ); ?>[holidays_new][label]"
								   value=""
								   placeholder="New holiday label"
								   style="width:100%;" />
							<input type="hidden"
								   name="<?php echo esc_attr( $this->option_name ); ?>[holidays_new][id]"
								   value="" />
						</td>
						<td>
							<input type="text"
								   name="<?php echo esc_attr( $this->option_name ); ?>[holidays_new][start]"
								   value=""
								   placeholder="MM-DD"
								   style="width:100%;" />
						</td>
						<td>
							<input type="text"
								   name="<?php echo esc_attr( $this->option_name ); ?>[holidays_new][end]"
								   value=""
								   placeholder="MM-DD"
								   style="width:100%;" />
						</td>
						<td>
							<!-- Advanced fields for new row use defaults, will be editable via modal after first save -->
							<span class="description">Save, then click Edit.</span>
						</td>
						<td>&nbsp;</td>
					</tr>
					</tbody>
				</table>
			
				<div id="ahs-admin-modal" class="ahs-admin-modal" style="display:none;">
					<div class="ahs-admin-modal-backdrop"></div>
					<div class="ahs-admin-modal-dialog">
						<button type="button" class="ahs-admin-modal-close" aria-label="Close">×</button>
						<h2 class="ahs-admin-modal-title">Edit holiday</h2>

						<input type="hidden" id="ahs_modal_row_index" value="" />
						<input type="hidden" id="ahs_modal_effect" value="snow" />
							<div class="ahs-modal-form">

							<!-- Message text -->
							<div class="ahs-modal-block">
								<h3 class="ahs-modal-block-title">Message text</h3>
								<textarea id="ahs_modal_message" rows="3"></textarea>
							</div>

							<!-- Font settings -->
							<div class="ahs-modal-block">
								<h3 class="ahs-modal-block-title">Font settings</h3>

								<div class="ahs-grid-row ahs-grid-font">
									<div class="ahs-grid-cell">
										<label for="ahs_modal_font_family"><strong>Font family</strong></label><br/>
										<select id="ahs_modal_font_family">
											<option value="">Theme default</option>
											<option value="Arial, sans-serif">Arial</option>
											<option value="'Helvetica Neue', Helvetica, Arial, sans-serif">Helvetica / System</option>
											<option value="'Times New Roman', serif">Times New Roman</option>
											<option value="'Georgia', serif">Georgia</option>
											<option value="'Courier New', monospace">Courier New</option>
										</select>
									</div>

									<div class="ahs-grid-cell">
										<label for="ahs_modal_font_size"><strong>Font size</strong></label><br/>
										<input type="number" id="ahs_modal_font_size" />
									</div>

									<div class="ahs-grid-cell">
										<label for="ahs_modal_font_style"><strong>Font style</strong></label><br/>
										<select id="ahs_modal_font_style">
											<option value="normal">Normal</option>
											<option value="bold">Bold</option>
											<option value="italic">Italic</option>
											<option value="bold_italic">Bold + Italic</option>
										</select>
									</div>

									<div class="ahs-grid-cell">
										<label for="ahs_modal_font_color"><strong>Font color</strong></label><br/>
										<input type="text" id="ahs_modal_font_color" class="ahs-color-picker" />
									</div>
								</div>
							</div>

							<!-- Text and layout -->
							<div class="ahs-modal-block">
								<h3 class="ahs-modal-block-title">Text and layout</h3>

								<div class="ahs-grid-row">
									<div class="ahs-grid-cell">
										<label for="ahs_modal_text_h"><strong>Text H</strong></label><br/>
										<select id="ahs_modal_text_h">
											<option value="left">Left</option>
											<option value="center">Center</option>
											<option value="right">Right</option>
										</select>
									</div>

									<div class="ahs-grid-cell">
										<label for="ahs_modal_text_v"><strong>Text V</strong></label><br/>
										<select id="ahs_modal_text_v">
											<option value="top">Top</option>
											<option value="middle">Middle</option>
											<option value="bottom">Bottom</option>
										</select>
									</div>

									<div class="ahs-grid-cell">
										<label for="ahs_modal_text_offset_x"><strong>Text offset X</strong></label><br/>
										<input type="number" id="ahs_modal_text_offset_x" step="1" />
									</div>

									<div class="ahs-grid-cell">
										<label for="ahs_modal_text_offset_y"><strong>Text offset Y</strong></label><br/>
										<input type="number" id="ahs_modal_text_offset_y" step="1" />
									</div>
								</div>

								<div class="ahs-grid-row">
									<div class="ahs-grid-cell ahs-grid-cell-wide">
										<label><strong>Location</strong></label><br/>
										<div class="ahs-radio-row" id="ahs_modal_location_radios">
											<label><input type="radio" name="ahs_modal_location" value="top" checked> Top (above header)</label>
											<label><input type="radio" name="ahs_modal_location" value="bottom"> Bottom (above footer)</label>
											<label><input type="radio" name="ahs_modal_location" value="modal"> Modal popup</label>
										</div>
									</div>
								</div>
							</div>

							<!-- Background -->
							<div class="ahs-modal-block">
								<h3 class="ahs-modal-block-title">Background</h3>

								<div class="ahs-grid-row">
									<div class="ahs-grid-cell">
										<label for="ahs_modal_bg_color"><strong>Color</strong></label><br/>
										<input type="text" id="ahs_modal_bg_color" class="ahs-color" />
									</div>

									<div class="ahs-grid-cell">
										<label><strong>Image</strong></label><br/>
										<button type="button" class="button" id="ahs_modal_bg_image_btn">Select image</button>
										<input type="hidden" id="ahs_modal_bg_image" value="" />
									</div>
								</div>

								<div class="ahs-grid-row">
									<div class="ahs-grid-cell">
										<label for="ahs_modal_bg_repeat"><strong>Repeat</strong></label><br/>
										<select id="ahs_modal_bg_repeat">
											<option value="no-repeat">No repeat</option>
											<option value="repeat">Tile</option>
											<option value="repeat-x">Repeat X</option>
											<option value="repeat-y">Repeat Y</option>
										</select>
									</div>

									<div class="ahs-grid-cell">
										<label for="ahs_modal_bg_opacity"><strong>Image transparency</strong></label><br/>
										<input type="number" id="ahs_modal_bg_opacity" min="0" max="100" step="1" />
									</div>

									<div class="ahs-grid-cell">
										<label for="ahs_modal_bg_position"><strong>Position</strong></label><br/>
										<select id="ahs_modal_bg_position">
											<option value="center center">Center</option>
											<option value="top left">Top left</option>
											<option value="top center">Top center</option>
											<option value="top right">Top right</option>
											<option value="center left">Center left</option>
											<option value="center right">Center right</option>
											<option value="bottom left">Bottom left</option>
											<option value="bottom center">Bottom center</option>
											<option value="bottom right">Bottom right</option>
										</select>
									</div>

									<div class="ahs-grid-cell">
										<label for="ahs_modal_bg_size"><strong>Size</strong></label><br/>
										<select id="ahs_modal_bg_size">
											<option value="cover">Cover</option>
											<option value="contain">Contain</option>
											<option value="auto">Auto</option>
										</select>
									</div>

									<div class="ahs-grid-cell">
										<label for="ahs_modal_bg_offset_x"><strong>Image offset X</strong></label><br/>
										<input type="number" id="ahs_modal_bg_offset_x" step="1" />
									</div>

									<div class="ahs-grid-cell">
										<label for="ahs_modal_bg_offset_y"><strong>Image offset Y</strong></label><br/>
										<input type="number" id="ahs_modal_bg_offset_y" step="1" />
									</div>
								</div>
							</div>

							<!-- General -->
							<div class="ahs-modal-block">
								<h3 class="ahs-modal-block-title">General</h3>

								<div class="ahs-grid-row">
									<div class="ahs-grid-cell">
										<label for="ahs_modal_height"><strong>Banner height</strong></label><br/>
										<input type="number" id="ahs_modal_height" />
										<div class="description">px (minimum height for the bar)</div>
									</div>

									<div class="ahs-grid-cell">
										<label for="ahs_modal_overlap"><strong>Overlap value</strong></label><br/>
										<input type="number" id="ahs_modal_overlap" />
										<div class="description">px (positive = up; negative = down)</div>
									</div>
								</div>
							</div>

							<!-- Falling groups -->
							<div class="ahs-modal-block">
								<h3 class="ahs-modal-block-title">Falling Objects</h3>

								<div class="cr-groups-toolbar">
									<span class="description">
										Create 1 or multiple groups (ex: 1× flag + 3× leaves). If a group’s items list is empty, built-in defaults are used.
									</span>
								</div>

								<div id="cr_groups_container" class="cr-groups-container"></div>

								<template id="cr_group_tpl">
									<div class="cr-group-card" data-group="">
										<div class="cr-group-grid">
											<label>
												<span class="cr-label">Type</span>
												<select class="cr-group-type">
													<option value="snow">Snow</option>
													<option value="leaves">Leaves</option>
													<option value="flag">Flag</option>
													<option value="custom">Custom</option>
												</select>
											</label>

											<label class="cr-only-custom">
												<span class="cr-label">Custom assets</span>
												<select class="cr-group-assets"></select>
											</label>

											<label>
												<span class="cr-label">Count</span>
												<input type="number" class="cr-group-count cr-num-sm" min="1" step="1" value="80" />
											</label>

											<button type="button" class="button-link-delete cr-group-remove" title="Remove group">Remove</button>
										</div>
									</div>
								</template>
								<button type="button" class="button button-secondary" id="cr_add_group">Add group</button>
							</div>

						</div>

						<p class="submit">
							<button type="button" class="button button-primary" id="ahs_modal_save">Save</button>
							<button type="button" class="button" id="ahs_modal_cancel">Cancel</button>
						</p>
					</div>
				</div>

                    <p class="description">
                        To remove a holiday, tick the “Delete” column and click “Save Changes”.
                        To add a holiday, fill the bottom row and click “Save Changes”.
                    </p>   

                    <?php submit_button(); ?>
                </form>

<style>
.cr-panel { margin: 12px 0 18px; }
.cr-grid {
    display: grid;
    grid-template-columns: 1fr;
    gap: 12px;
}
@media (min-width: 960px) {
    .cr-grid { grid-template-columns: 1fr 1fr; }
    .cr-field-wide { grid-column: 1 / -1; }
}

.cr-field {
    background: #fff;
    border: 1px solid rgba(90, 50, 180, 0.14);
    border-radius: 16px;
    padding: 12px 14px;
    box-shadow: 0 10px 22px rgba(38, 17, 84, 0.06);
}
	#cr_global_note_wrap,
#cr_debug_wrap {
    min-height: 180px;
}

.cr-textarea-fixed {
    width: 100%;
    height: 110px;
    resize: none;
}
.cr-debug-date.is-disabled {
    opacity: 0.55;
    pointer-events: none;
}

.cr-field-top {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 12px;
}

.cr-help {
    margin-top: 6px;
    font-size: 12px;
    color: rgba(40, 20, 90, 0.70);
}

.cr-actions {
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
    align-items: center;
}

.cr-status {
    font-size: 12px;
    color: rgba(40, 20, 90, 0.70);
}
.cr-inline-actions {
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
    align-items: center;
    margin-top: 8px;
}

.cr-textarea-fixed {
    width: 100%;
    height: 84px;
    min-height: 84px;
    resize: none;
}

.cr-debug-date {
    margin-top: 10px;
    padding-top: 10px;
    border-top: 1px dashed rgba(40, 20, 90, 0.18);
}

.cr-switch {
    position: relative;
    width: 46px;
    height: 26px;
    display: inline-block;
}
.cr-switch input {
    opacity: 0;
    width: 0;
    height: 0;
}
.cr-slider{
    position: absolute;
    inset: 0;
    cursor: pointer;
    background: rgba(40, 20, 90, 0.18);
    border-radius: 999px;
    transition: 160ms ease;
    box-shadow: inset 0 0 0 1px rgba(40, 20, 90, 0.18);
}
.cr-slider:before{
    content: "";
    position: absolute;
    left: 3px;
    top: 3px;
    width: 20px;
    height: 20px;
    border-radius: 50%;
    background: #fff;
    transition: 160ms ease;
    box-shadow: 0 6px 14px rgba(0,0,0,0.18);
}
.cr-switch input:checked + .cr-slider{
    background: rgba(120, 64, 230, 0.92);
}
.cr-switch input:checked + .cr-slider:before{
    transform: translateX(20px);
}

.wrap > h1 { margin-bottom: 10px; }

.ahs-admin-shell { max-width: 1280px; }

.widefat.ahs-holiday-table,
.widefat.ahs-assets-table {
    border-radius: 18px;
    overflow: hidden;
    border: 1px solid rgba(90, 50, 180, 0.12);
    box-shadow: 0 12px 26px rgba(38, 17, 84, 0.10);
}

.widefat.ahs-holiday-table thead th,
.widefat.ahs-assets-table thead th {
    background: linear-gradient(90deg, rgba(120, 64, 230, 0.95), rgba(74, 96, 245, 0.95));
    color: #ffffff;
    border-bottom: 0;
    font-weight: 700;
}

.widefat.ahs-holiday-table tbody tr:nth-child(odd),
.widefat.ahs-assets-table tbody tr:nth-child(odd) {
    background: rgba(246, 244, 255, 0.75);
}

.ahs-grid-row{
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(160px, 1fr));
    gap: 12px;
    align-items: start;
}

.ahs-grid-cell{
    min-width: 0;       
    overflow: visible;  
}

.ahs-grid-cell-wide{
    grid-column: 1 / -1; 
}

.ahs-grid-row.ahs-grid-font {
    display: grid;
    grid-template-columns: minmax(220px, 1fr) 90px 160px minmax(220px, 1fr);
    column-gap: 10px; 
    row-gap: 12px;
    align-items: end;
}

.ahs-grid-row.ahs-grid-font #ahs_modal_font_size{
    width: 100% !important;
    max-width: 90px;
}

@media (max-width: 782px) {
    .ahs-grid-row.ahs-grid-font {
        grid-template-columns: 1fr 1fr;
    }
}
@media (max-width: 520px) {
    .ahs-grid-row.ahs-grid-font {
        grid-template-columns: 1fr;
    }
}

.ahs-grid-cell-wide {
    flex: 2 1 380px;
    min-width: 260px;
}
.ahs-radio-row {
    display: flex;
    flex-wrap: wrap;
    gap: 14px;
    margin-top: 6px;
}
.ahs-radio-row label {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    font-weight: 600;
}

.ahs-admin-modal { position: fixed; inset: 0; z-index: 99999; display: block; }
.ahs-admin-modal-backdrop { position: absolute; inset: 0; background: rgba(20, 10, 40, 0.55); }

.ahs-admin-modal-dialog {
    position: relative;
    margin: 30px auto;
    background: #ffffff;
    width: 92%;
    max-width: 760px;
    max-height: 88vh;
    overflow: auto;
    padding: 16px 16px 12px;
    border-radius: 18px;
    box-shadow: 0 20px 60px rgba(0,0,0,0.30);
    border: 1px solid rgba(90, 50, 180, 0.16);
}

.ahs-modal-form{
    display: flex;
    flex-direction: column;
    gap: 14px;
}

.ahs-modal-block{
    background: rgba(246, 244, 255, 0.65);
    border: 1px solid rgba(90, 50, 180, 0.12);
    border-radius: 16px;
    padding: 12px 14px;
}

.ahs-modal-block-title{
    margin: 0 0 10px;
    text-align: center;
    font-weight: 800;
    color: rgba(40, 20, 90, 0.85);
}

.ahs-admin-modal-dialog .ahs-modal-form input[type="text"],
.ahs-admin-modal-dialog .ahs-modal-form input[type="number"],
.ahs-admin-modal-dialog .ahs-modal-form select,
.ahs-admin-modal-dialog .ahs-modal-form textarea{
    width: 100%;
    max-width: none;
    box-sizing: border-box;
}

.ahs-admin-modal-dialog input[type="text"],
.ahs-admin-modal-dialog input[type="number"],
.ahs-admin-modal-dialog select,
.ahs-admin-modal-dialog textarea{
    width: 100%;
    max-width: none;
    box-sizing: border-box;
}

.ahs-admin-modal-dialog .wp-picker-container{
    width: auto !important;
    display: inline-flex !important;
    align-items: center;
}

.ahs-admin-modal-dialog .wp-picker-container .wp-color-result.button{
    width: 220px !important;     
    max-width: 100% !important;
}

.ahs-admin-modal-dialog .wp-picker-container .wp-color-result.button .wp-color-result-text{
    width: 100%;
}

.ahs-admin-modal-dialog .wp-picker-holder{
    left: 0;
}

.ahs-admin-modal-dialog input[type="number"][style*="width:80px"]{
    width: 80px !important;
}

.ahs-admin-modal-dialog .wp-picker-container{
    position: relative;
    display: block;
    width: 100%;
    margin: 0;
}

.ahs-admin-modal-dialog .wp-picker-container .wp-color-result.button{
    width: 100%;
    max-width: none;
    box-sizing: border-box;
    margin: 0;
    height: 40px;
    line-height: 38px;
    border-radius: 12px;
}

.ahs-admin-modal-dialog .wp-picker-container .wp-color-result.button{
    display: flex;
    align-items: center;
    justify-content: center;
    padding: 0 !important;
    overflow: hidden; 
    position: relative;
}

.ahs-admin-modal-dialog .wp-picker-container .wp-color-result.button .wp-color-result-text{
    flex: 1;
    width: 100%;
    height: 100%;
    display: flex;
    align-items: center;
    justify-content: center;

    background: #fff;
    margin: 0;
    border-left: 0; 
    border-radius: 12px; 
}
	
.ahs-admin-modal-dialog .wp-picker-container .wp-color-result.button::before{
    content: "";
    position: absolute;
    left: 8px;
    top: 8px;
    width: 24px;
    height: 24px;
    border-radius: 8px;
    background: inherit;
    box-shadow: inset 0 0 0 1px rgba(0,0,0,.25);
}
	
.ahs-admin-modal-dialog .wp-picker-container .wp-picker-input-wrap,
.ahs-admin-modal-dialog .wp-picker-container .wp-picker-clear,
.ahs-admin-modal-dialog .wp-picker-container .wp-picker-default{
    display: none !important;
}

.ahs-admin-modal-dialog .wp-picker-container:not(.wp-picker-active) .wp-picker-holder{
    display: none !important;
}
	
.ahs-admin-modal-dialog .wp-picker-container.wp-picker-active .wp-picker-holder{
    display: block !important;
    position: absolute;
    top: calc(100% + 6px);
    left: 0;
    z-index: 100000;
}

.ahs-admin-modal-dialog .button,
.ahs-admin-modal-dialog .button-primary,
.ahs-admin-modal-dialog .button-secondary{
    border-radius: 10px;
    min-height: 34px;
    line-height: 32px;
}

.celeb-right-admin .button,
.celeb-right-admin .button-secondary,
.celeb-right-admin .button-primary,
.celeb-right-admin .ahs-admin-modal-dialog .button,
.celeb-right-admin .ahs-admin-modal-dialog .button-secondary,
.celeb-right-admin .ahs-admin-modal-dialog .button-primary{
    appearance: none;
    background: #ffffff !important;
    color: rgba(40, 20, 90, 0.88) !important;

    border: 1px solid rgba(40, 20, 90, 0.18) !important;
    border-radius: 12px !important;

    box-shadow: 0 10px 22px rgba(38, 17, 84, 0.06) !important;

    padding: 0 14px !important;
    min-height: 36px !important;
    line-height: 34px !important;

    font-weight: 700 !important;
    text-decoration: none !important;

    transition: background-color 160ms ease, border-color 160ms ease, box-shadow 160ms ease, transform 120ms ease;
}

.celeb-right-admin .button:hover,
.celeb-right-admin .button-secondary:hover,
.celeb-right-admin .button-primary:hover,
.celeb-right-admin .ahs-admin-modal-dialog .button:hover,
.celeb-right-admin .ahs-admin-modal-dialog .button-secondary:hover,
.celeb-right-admin .ahs-admin-modal-dialog .button-primary:hover{
    background: rgba(246, 244, 255, 0.90) !important;
    border-color: rgba(120, 64, 230, 0.45) !important;
}

.celeb-right-admin .button:focus,
.celeb-right-admin .button-secondary:focus,
.celeb-right-admin .button-primary:focus,
.celeb-right-admin .ahs-admin-modal-dialog .button:focus,
.celeb-right-admin .ahs-admin-modal-dialog .button-secondary:focus,
.celeb-right-admin .ahs-admin-modal-dialog .button-primary:focus{
    outline: none !important;
    box-shadow: 0 0 0 3px rgba(120, 64, 230, 0.22), 0 10px 22px rgba(38, 17, 84, 0.06) !important;
}

.celeb-right-admin .button:active,
.celeb-right-admin .button-secondary:active,
.celeb-right-admin .button-primary:active,
.celeb-right-admin .ahs-admin-modal-dialog .button:active,
.celeb-right-admin .ahs-admin-modal-dialog .button-secondary:active,
.celeb-right-admin .ahs-admin-modal-dialog .button-primary:active{
    transform: translateY(1px);
}

.celeb-right-admin .button-primary,
.celeb-right-admin .ahs-admin-modal-dialog .button-primary{
    border-color: rgba(120, 64, 230, 0.55) !important;
    color: rgba(120, 64, 230, 0.98) !important;
}

.celeb-right-admin .button-primary:hover,
.celeb-right-admin .ahs-admin-modal-dialog .button-primary:hover{
    background: rgba(120, 64, 230, 0.08) !important;
    border-color: rgba(120, 64, 230, 0.70) !important;
}

.celeb-right-admin .button-link-delete,
.celeb-right-admin .ahs-admin-modal-dialog .button-link-delete{
    display: inline-flex !important;
    align-items: center;
    justify-content: center;

    background: #ffffff !important;
    color: #b01f29 !important;

    border: 1px solid rgba(176, 31, 41, 0.45) !important;
    border-radius: 12px !important;

    padding: 0 14px !important;
    min-height: 36px !important;
    line-height: 34px !important;

    box-shadow: 0 10px 22px rgba(38, 17, 84, 0.06) !important;
    text-decoration: none !important;

    transition: background-color 160ms ease, border-color 160ms ease, box-shadow 160ms ease, transform 120ms ease;
}

.celeb-right-admin .button-link-delete:hover,
.celeb-right-admin .ahs-admin-modal-dialog .button-link-delete:hover{
    background: rgba(176, 31, 41, 0.08) !important;
    border-color: rgba(176, 31, 41, 0.65) !important;
}

.celeb-right-admin .button-link-delete:focus,
.celeb-right-admin .ahs-admin-modal-dialog .button-link-delete:focus{
    outline: none !important;
    box-shadow: 0 0 0 3px rgba(176, 31, 41, 0.18), 0 10px 22px rgba(38, 17, 84, 0.06) !important;
}

.celeb-right-admin .cr-actions .button,
.celeb-right-admin .cr-actions--compact .button{
    margin: 0 !important;
}

.ahs-admin-modal-table th,
.ahs-admin-modal-table td {
    padding: 8px 10px !important;
}

.ahs-admin-modal-title {
    margin: 0 0 12px;
    padding: 12px 14px;
    border-radius: 14px;
    color: #ffffff;
    background: linear-gradient(90deg, rgba(120, 64, 230, 0.98), rgba(250, 86, 180, 0.98));
}

.ahs-admin-modal-close {
    position: absolute;
    right: 14px;
    top: 10px;
    width: 36px;
    height: 36px;
    border-radius: 10px;
    border: 1px solid rgba(255,255,255,0.35);
    background: rgba(255,255,255,0.12);
    color: #ffffff;
    cursor: pointer;
    font-size: 22px;
    line-height: 34px;
}
	
.cr-groups-toolbar {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 10px;
    flex-wrap: wrap;
    margin-bottom: 10px;
}

.cr-groups-container {
    display: grid;
    grid-template-columns: 1fr;
    gap: 12px;
}

.cr-group-card {
    border-radius: 16px;
    border: 1px solid rgba(90, 50, 180, 0.12);
    background: rgba(246, 244, 255, 0.65);
    padding: 12px;
}

.cr-group-head {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 10px;
    margin-bottom: 10px;
}

.cr-group-title { font-weight: 800; color: rgba(40, 20, 90, 0.85); }

.cr-group-grid{
    display: grid;
    grid-template-columns: 1fr;
    gap: 12px;
    align-items: end;
    margin: 0;
}

@media (min-width: 780px){
    .cr-group-grid{
        grid-template-columns: 180px 1fr 120px auto;
        column-gap: 12px;
        row-gap: 10px;
    }
    .cr-group-remove{
        justify-self: end;
        align-self: end;
        margin: 0;
    }
}
	
.cr-group-grid select,
.cr-group-grid input[type="number"]{
    width: 100%;
    max-width: none;
    box-sizing: border-box;
}

.cr-num-sm{
    width: 90px;
    max-width: 90px;
}

.cr-group-card:not(.is-custom) .cr-only-custom {
    opacity: 0.45;
}

.cr-group-card:not(.is-custom) .cr-only-custom select {
    background: #f3f3f3;
}
	
#cr_add_group{
    float: right;
    margin-top: 10px; 
}
	
#cr_add_group::after{
    content:"";
    display:block;
    clear: both;
}

.cr-group-assets:disabled {
    cursor: not-allowed;
}

.cr-label {
    display: block;
    font-size: 12px;
    font-weight: 700;
    margin-bottom: 4px;
    color: rgba(40, 20, 90, 0.75);
}

.cr-group-add-row { display: flex; gap: 8px; align-items: center; flex-wrap: wrap; }
.cr-group-asset-pick { min-width: 260px; }

.cr-items { display: flex; gap: 8px; flex-wrap: wrap; padding-top: 6px; }

.cr-chip {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 6px 10px;
    border-radius: 999px;
    background: #ffffff;
    border: 1px solid rgba(90, 50, 180, 0.14);
    box-shadow: 0 8px 18px rgba(38, 17, 84, 0.08);
}
.cr-chip img { width: 22px; height: 22px; object-fit: contain; }

.cr-chip button {
    border: none;
    background: transparent;
    color: rgba(40, 20, 90, 0.7);
    cursor: pointer;
    font-size: 16px;
    line-height: 1;
    padding: 0;
}

@media (max-width: 782px) {
    .ahs-admin-modal-dialog { margin: 10px auto; }
}
</style>



<script>
(function($) {
    var mediaFrameBg = null;
var mediaFrameAsset = null;

	window.CR_EXPORT_DATA = <?php echo wp_json_encode( $holidays ); ?>;

    function findHiddenRow(rowIndex) {
        return $('.ahs-hidden-row[data-row="' + rowIndex + '"]');
    }

    function findMainRow(rowIndex) {
        return $('.ahs-holiday-row[data-row="' + rowIndex + '"]');
    }
	
	function escHtml(str) {
    return String(str || '').replace(/[&<>"']/g, function (s) {
        return ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'})[s];
    });
}

function getAllAssets() {
    var list = [];

    $('#ahs_assets_table tbody tr.ahs-asset-row').each(function(){
        var name = String($(this).find('.ahs-asset-name').val() || '').trim();
        var url  = String($(this).find('.ahs-asset-url').val() || '').trim();
        if (!url) return;

        if (!name) {
            try {
                var u = new URL(url);
                name = (u.pathname.split('/').pop() || url);
            } catch (e) {
                name = url;
            }
        }

        list.push({ name: name, url: url });
    });

    var seen = {};
    return list.filter(function(a){
        if (!a.url) return false;
        if (seen[a.url]) return false;
        seen[a.url] = true;
        return true;
    });
}

function renderAssetPick($select) {
    var current = $select.val();
    var assets = getAllAssets();

    var html = '<option value="">Select…</option>';
    assets.forEach(function(a){
        html += '<option value="' + escHtml(a.url) + '">' + escHtml(a.name) + '</option>';
    });

    $select.html(html);

    if (current && current.length) {
        $select.val(current);
    }
}

function openMedia(onPick) {
    if (mediaFrameAsset) {
        mediaFrameAsset.off('select');
    }

    mediaFrameAsset = wp.media({
        title: 'Select asset',
        button: { text: 'Use this asset' },
        multiple: false
    });

    mediaFrameAsset.on('select', function() {
        var attachment = mediaFrameAsset.state().get('selection').first().toJSON();
        if (onPick) onPick(attachment.url);
    });

    mediaFrameAsset.open();
}


function addChip($itemsWrap, url) {
    if (!url) return;

    var exists = false;
    $itemsWrap.find('.cr-chip').each(function(){
        if ($(this).data('url') === url) exists = true;
    });
    if (exists) return;

    var $chip = $('<span class="cr-chip"></span>').attr('data-url', url).data('url', url);
    var $img  = $('<img />').attr('src', url).attr('alt', '');
    var $txt  = $('<span></span>').text(url.split('/').pop());
    var $btn  = $('<button type="button" aria-label="Remove">×</button>');
    $btn.on('click', function(){ $chip.remove(); });

    $chip.append($img, $txt, $btn);
    $itemsWrap.append($chip);
}

function addGroupCard(groupData) {
    var tpl = document.getElementById('cr_group_tpl');
    if (!tpl) return null;

    var node = tpl.content.firstElementChild.cloneNode(true);
    var $card = $(node);
    $('#cr_groups_container').append($card);

    renderAssetPick($card.find('.cr-group-assets'));

    var type  = (groupData && groupData.type) ? groupData.type : 'leaves';
    var count = (groupData && groupData.count) ? parseInt(groupData.count, 10) : 80;
    if (!count || count < 1) count = 80;

    $card.find('.cr-group-type').val(type);
    $card.find('.cr-group-count').val(count);

    var $items = $card.find('.cr-items');

    function setTypeUI() {
    var t = $card.find('.cr-group-type').val() || 'leaves';
    var isCustom = (t === 'custom');

    $card.toggleClass('is-custom', isCustom);

    $card.find('.cr-group-assets').prop('disabled', !isCustom);

    if (!isCustom) {
        $card.find('.cr-group-assets').val([]);
    }
}

    setTypeUI();

	var items = (groupData && Array.isArray(groupData.items)) ? groupData.items : [];
	if (type === 'custom' && items.length) {
		$card.find('.cr-group-assets').val(items);
	}

    $card.find('.cr-group-type').on('change', function(){
        setTypeUI();
    });

    $card.find('.cr-group-remove').on('click', function(e){
        e.preventDefault();
        $card.remove();
    });


    return $card;
}


function setGroupsUI(groups) {
    $('#cr_groups_container').empty();
    if (!Array.isArray(groups) || groups.length === 0) {
        var preset = $('#ahs_modal_effect').val() || 'leaves';
        if (preset === 'none') return;
        addGroupCard({ type: preset, count: preset === 'snow' ? 120 : (preset === 'flag' ? 60 : 80), items: [] });
        return;
    }
    groups.forEach(function(g){ addGroupCard(g); });
}

function collectGroupsUI() {
    var groups = [];
    $('#cr_groups_container .cr-group-card').each(function(){
        var $g = $(this);
        var type = $g.find('.cr-group-type').val() || 'leaves';
        var count = parseInt($g.find('.cr-group-count').val(), 10);
        if (!count || count < 1) count = 1;

        var items = [];
		if (type === 'custom') {
    var v = $g.find('.cr-group-assets').val();

    if (Array.isArray(v)) {
        items = v.filter(function(x){ return !!x; });
    } else if (typeof v === 'string') {
        v = v.trim();
        items = v ? [v] : [];
    }
}

		groups.push({ type: type, count: count, items: items });

    });

    return groups;
}


    function openModal(rowIndex) {
        var $hidden = findHiddenRow(rowIndex);
        var $main   = findMainRow(rowIndex);

        if (!$hidden.length || !$main.length) {
            return;
        }

        $('#ahs_modal_row_index').val(rowIndex);

        $('#ahs_modal_message').val($hidden.find('.ahs-field-message').val() || '');
        $('#ahs_modal_effect').val($hidden.find('.ahs-field-effect').val() || 'snow');

		try {
			var groupsRaw = $hidden.find('.ahs-field-effect-groups').val() || '[]';
			var groupsObj = JSON.parse(groupsRaw);
			setGroupsUI(groupsObj);
		} catch (err) {
			setGroupsUI([]);
		}

        var locVal = $hidden.find('.ahs-field-location').val() || 'top';
		$('input[name="ahs_modal_location"][value="' + locVal + '"]').prop('checked', true);


        var colorVal = $hidden.find('.ahs-field-bg-color').val() || '';

        if ($.fn.wpColorPicker) {
            
            $('#ahs_modal_bg_color').wpColorPicker('color', colorVal);
        } else {
            $('#ahs_modal_bg_color').val(colorVal);
        }

        $('#ahs_modal_bg_image').val($hidden.find('.ahs-field-bg-image').val() || '');
        $('#ahs_modal_bg_repeat').val($hidden.find('.ahs-field-bg-repeat').val() || 'no-repeat');
        $('#ahs_modal_bg_position').val($hidden.find('.ahs-field-bg-position').val() || 'center top');
        $('#ahs_modal_bg_size').val($hidden.find('.ahs-field-bg-size').val() || 'cover');
        $('#ahs_modal_height').val($hidden.find('.ahs-field-banner-height').val() || '');
        $('#ahs_modal_overlap').val($hidden.find('.ahs-field-overlap').val() || '');
        $('#ahs_modal_bg_opacity').val($hidden.find('.ahs-field-bg-opacity').val() || '');
		$('#ahs_modal_bg_offset_x').val($hidden.find('.ahs-field-bg-offset-x').val() || '');
		$('#ahs_modal_bg_offset_y').val($hidden.find('.ahs-field-bg-offset-y').val() || '');
        $('#ahs_modal_text_h').val($main.find('.ahs-field-text-h').val() || 'center');
        $('#ahs_modal_text_v').val($main.find('.ahs-field-text-v').val() || 'middle');
        $('#ahs_modal_text_offset_x').val($main.find('.ahs-field-text-offset-x').val() || '');
		$('#ahs_modal_text_offset_y').val($main.find('.ahs-field-text-offset-y').val() || '');
        $('#ahs_modal_font_family').val($hidden.find('.ahs-field-font-family').val() || '');
        $('#ahs_modal_font_size').val($hidden.find('.ahs-field-font-size').val() || '');
        $('#ahs_modal_font_style').val($hidden.find('.ahs-field-font-style').val() || 'normal');

        var fontColor = $hidden.find('.ahs-field-font-color').val() || '';
        if ($.fn.wpColorPicker) {
            $('#ahs_modal_font_color').wpColorPicker('color', fontColor);
        } else {
            $('#ahs_modal_font_color').val(fontColor);
        }

		
        $('#ahs-admin-modal').show();
    }

    function closeModal() {
    closePickers();
    $('#ahs-admin-modal').hide();
}

    function saveModal() {
    var rowIndex = $('#ahs_modal_row_index').val();

    var $hidden = findHiddenRow(rowIndex);
    var $main   = findMainRow(rowIndex);

    if (!$hidden.length || !$main.length) {
        closeModal();
        return;
    }

    var groupsToSave = collectGroupsUI();
    $hidden.find('.ahs-field-effect-groups').val(JSON.stringify(groupsToSave));

    var legacy = (groupsToSave.length ? groupsToSave[0].type : 'none');
    if (legacy === 'custom') legacy = 'none';
    $('#ahs_modal_effect').val(legacy);
    $hidden.find('.ahs-field-effect').val(legacy);

    $hidden.find('.ahs-field-message').val($('#ahs_modal_message').val());
    $hidden.find('.ahs-field-location').val(($('input[name="ahs_modal_location"]:checked').val() || 'top'));
    $hidden.find('.ahs-field-bg-color').val($('#ahs_modal_bg_color').val());
    $hidden.find('.ahs-field-bg-image').val($('#ahs_modal_bg_image').val());
    $hidden.find('.ahs-field-bg-repeat').val($('#ahs_modal_bg_repeat').val());
    $hidden.find('.ahs-field-bg-position').val($('#ahs_modal_bg_position').val());
    $hidden.find('.ahs-field-bg-size').val($('#ahs_modal_bg_size').val());
    $hidden.find('.ahs-field-banner-height').val($('#ahs_modal_height').val());
    $hidden.find('.ahs-field-overlap').val($('#ahs_modal_overlap').val());
    $hidden.find('.ahs-field-bg-opacity').val($('#ahs_modal_bg_opacity').val());
    $hidden.find('.ahs-field-bg-offset-x').val($('#ahs_modal_bg_offset_x').val());
    $hidden.find('.ahs-field-bg-offset-y').val($('#ahs_modal_bg_offset_y').val());
    $hidden.find('.ahs-field-font-family').val($('#ahs_modal_font_family').val());
    $hidden.find('.ahs-field-font-size').val($('#ahs_modal_font_size').val());
    $hidden.find('.ahs-field-font-style').val($('#ahs_modal_font_style').val());
    $hidden.find('.ahs-field-font-color').val($('#ahs_modal_font_color').val());
    $main.find('.ahs-field-text-h').val($('#ahs_modal_text_h').val());
    $main.find('.ahs-field-text-v').val($('#ahs_modal_text_v').val());
    $main.find('.ahs-field-text-offset-x').val($('#ahs_modal_text_offset_x').val());
	$main.find('.ahs-field-text-offset-y').val($('#ahs_modal_text_offset_y').val());
    closeModal();
}

        function closePickers() {
 
    if ($.fn.iris) {
        $('#ahs_modal_bg_color, #ahs_modal_font_color').iris('hide');
    }
    $('#ahs-admin-modal .wp-picker-container').removeClass('wp-picker-active');
}

function initColorPicker() {
    if (typeof $.fn.wpColorPicker !== 'function') return;

    if (!$('#ahs_modal_bg_color').data('wpWpColorPicker')) {
        $('#ahs_modal_bg_color').wpColorPicker({ hide: true });
    }
    if (!$('#ahs_modal_font_color').data('wpWpColorPicker')) {
        $('#ahs_modal_font_color').wpColorPicker({ hide: true });
    }
}

    function initMediaButton() {
    $('#ahs_modal_bg_image_btn').on('click', function(e) {
        e.preventDefault();

        if (mediaFrameBg) {
            mediaFrameBg.off('select');
            mediaFrameBg.on('select', function() {
                var attachment = mediaFrameBg.state().get('selection').first().toJSON();
                $('#ahs_modal_bg_image').val(attachment.url || '');
            });
            mediaFrameBg.open();
            return;
        }

        mediaFrameBg = wp.media({
            title: 'Select background image',
            button: { text: 'Use this image' },
            multiple: false
        });

        mediaFrameBg.on('select', function() {
            var attachment = mediaFrameBg.state().get('selection').first().toJSON();
            $('#ahs_modal_bg_image').val(attachment.url || '');
        });

        mediaFrameBg.open();
    });
}

    $(function() {
        initColorPicker();
        initMediaButton();

function syncDebugUI(){
    var on = $('#ahs_debug_enabled').is(':checked');

    $('#cr_debug_date_wrap')
        .toggleClass('is-disabled', !on)
        .find('input, select, textarea, button')
        .prop('disabled', !on);
}
		
$('#ahs-admin-modal').on('mousedown', function(e){
    if ($(e.target).closest('.wp-picker-container').length) return;
    closePickers();
});

syncDebugUI();
$('#ahs_debug_enabled').on('change', syncDebugUI);

$('#cr_export_json').on('click', function(e){
    e.preventDefault();

    var data = window.CR_EXPORT_DATA || null;
    if (!data) {
        $('#cr_import_status').text('Export failed: no data found.');
        return;
    }

    var json = JSON.stringify(data, null, 2);
    var blob = new Blob([json], { type: 'application/json' });
    var url = URL.createObjectURL(blob);

    var a = document.createElement('a');
    a.href = url;
    a.download = 'celeb-right-holidays.json';
    document.body.appendChild(a);
    a.click();
    a.remove();

    setTimeout(function(){ URL.revokeObjectURL(url); }, 500);
});

$('#cr_import_json_choose').on('click', function(e){
    e.preventDefault();
    $('#cr_import_json_file').val('').trigger('click');
});

$('#cr_import_json_file').on('change', function(){
    var file = this.files && this.files[0] ? this.files[0] : null;
    if (!file) return;

    var reader = new FileReader();
    reader.onload = function(){
        try {
            var parsed = JSON.parse(reader.result);

            if (!Array.isArray(parsed)) {
                $('#cr_import_status').text('Import failed: JSON must be an array of holidays.');
                return;
            }

            $('#ahs_holidays_json_import').val(JSON.stringify(parsed));
            $('#cr_import_status').text('Import loaded. Click “Save Changes” to apply.');
        } catch (err) {
            $('#cr_import_status').text('Import failed: invalid JSON.');
        }
    };
    reader.readAsText(file);
});

$('#cr_import_json_clear').on('click', function(e){
    e.preventDefault();
    $('#ahs_holidays_json_import').val('');
    $('#cr_import_status').text('Import override cleared.');
});

		function renumberAssetRows() {
    $('#ahs_assets_table tbody tr.ahs-asset-row').each(function(i){
        $(this).attr('data-idx', i);
        $(this).find('input.ahs-asset-name').attr('name', '<?php echo esc_attr( $this->option_name ); ?>[assets_custom][' + i + '][name]');
        $(this).find('input.ahs-asset-url').attr('name', '<?php echo esc_attr( $this->option_name ); ?>[assets_custom][' + i + '][url]');
    });
}

$('#ahs_add_asset').on('click', function(e){
    e.preventDefault();
    var $tbody = $('#ahs_assets_table tbody');
    var i = $tbody.find('tr.ahs-asset-row').length;

    var row = ''
        + '<tr class="ahs-asset-row" data-idx="' + i + '">'
        + ' <td><input type="text" class="regular-text ahs-asset-name" name="<?php echo esc_attr( $this->option_name ); ?>[assets_custom][' + i + '][name]" value="" /></td>'
        + ' <td><input type="text" class="regular-text ahs-asset-url" style="width:100%;" name="<?php echo esc_attr( $this->option_name ); ?>[assets_custom][' + i + '][url]" value="" placeholder="https://example.com/asset.svg" /></td>'
        + ' <td><button type="button" class="button ahs-asset-upload">Upload</button></td>'
        + ' <td><button type="button" class="button-link-delete ahs-asset-remove">Remove</button></td>'
        + '</tr>';

    $tbody.append(row);

    $('#cr_groups_container .cr-group-assets').each(function(){
    renderAssetPick($(this));
});



});

$('#ahs_assets_table').on('click', '.ahs-asset-remove', function(e){
    e.preventDefault();
    $(this).closest('tr.ahs-asset-row').remove();
    renumberAssetRows();
    $('#cr_groups_container .cr-group-assets').each(function(){
    renderAssetPick($(this));
});



});

$('#ahs_assets_table').on('click', '.ahs-asset-upload', function(e){
    e.preventDefault();
    var $row = $(this).closest('tr.ahs-asset-row');
    openMedia(function(url){
        if (url) $row.find('.ahs-asset-url').val(url);
        $('#cr_groups_container .cr-group-assets').each(function(){
    renderAssetPick($(this));
});



    });
});

$('#ahs_assets_table').on('input', '.ahs-asset-name, .ahs-asset-url', function(){
   $('#cr_groups_container .cr-group-assets').each(function(){
    renderAssetPick($(this));
});

});

$('#cr_add_group').on('click', function(e){
    e.preventDefault();
    addGroupCard({ type: 'leaves', count: 80, items: [] });
});

$('#ahs_modal_effect').on('change', function(){
    var preset = $(this).val() || 'leaves';
    if (preset === 'none') {
        $('#cr_groups_container').empty();
        return;
    }
    setGroupsUI([{ type: preset, count: preset === 'snow' ? 120 : (preset === 'flag' ? 60 : 80), items: [] }]);
});

        $('.ahs-holiday-table').on('click', '.ahs-edit-btn', function(e) {
            e.preventDefault();
            var rowIndex = $(this).data('row');
            openModal(rowIndex);
        });

        $('#ahs_modal_cancel, .ahs-admin-modal-close, .ahs-admin-modal-backdrop').on('click', function(e) {
            e.preventDefault();
            closeModal();
        });

        $('#ahs_modal_save').on('click', function(e) {
            e.preventDefault();
            saveModal();
        });
    });
})(jQuery);
</script>
			   
            </div>
            <?php
        }

        public function determine_today_holiday() {
            if ( is_admin() ) {
                return;
            }
            $this->settings      = $this->get_settings();
            $this->today_holiday = $this->get_today_holiday();
        }

        private function get_today_holiday() {
            if ( empty( $this->settings['enabled'] ) ) {
                return null;
            }

            $use_debug = ! empty( $this->settings['debug_enabled'] ) && ! empty( $this->settings['debug_date'] );
            if ( $use_debug ) {
                $timestamp = strtotime( $this->settings['debug_date'] . ' 12:00:00' );
                if ( ! $timestamp ) {
                    $timestamp = current_time( 'timestamp' );
                }
            } else {
                $timestamp = current_time( 'timestamp' );
            }

            $today_md = date( 'm-d', $timestamp );
            $holidays = $this->settings['holidays'];

            foreach ( $holidays as $holiday ) {
                if ( isset( $holiday['enabled'] ) && ! $holiday['enabled'] ) {
                    continue;
                }

                $start = isset( $holiday['start'] ) ? $holiday['start'] : '';
                $end   = isset( $holiday['end'] ) ? $holiday['end'] : '';

                if ( '' === $start || '' === $end ) {
                    continue;
                }
				
                if ( $start <= $end ) {
                    if ( $today_md < $start || $today_md > $end ) {
                        continue;
                    }
                } else {
					
                    if ( $today_md < $start && $today_md > $end ) {
                        continue;
                    }
                }

                $id       = ! empty( $holiday['id'] ) ? sanitize_key( $holiday['id'] ) : sanitize_key( $holiday['label'] );
                $label    = ! empty( $holiday['label'] ) ? $holiday['label'] : $id;
                $message  = ! empty( $holiday['message'] ) ? $holiday['message'] : '';
                $effect   = ! empty( $holiday['effect'] ) ? $holiday['effect'] : 'snow';
                $location = ! empty( $holiday['location'] ) ? $holiday['location'] : 'top';

                $bg_color      = isset( $holiday['bg_color'] ) ? $holiday['bg_color'] : '';
                $bg_image      = isset( $holiday['bg_image'] ) ? $holiday['bg_image'] : '';
                $bg_repeat     = isset( $holiday['bg_repeat'] ) ? $holiday['bg_repeat'] : 'no-repeat';
                $bg_position   = isset( $holiday['bg_position'] ) ? $holiday['bg_position'] : 'center top';
                $bg_size       = isset( $holiday['bg_size'] ) ? $holiday['bg_size'] : 'cover';
                $banner_height = isset( $holiday['banner_height'] ) ? $holiday['banner_height'] : '';
				$bg_opacity = isset( $holiday['bg_opacity'] ) ? $holiday['bg_opacity'] : '';
				$overlap    = isset( $holiday['overlap'] ) ? $holiday['overlap'] : '';
				$bg_offset_x   = isset( $holiday['bg_offset_x'] ) ? $holiday['bg_offset_x'] : '';
				$bg_offset_y   = isset( $holiday['bg_offset_y'] ) ? $holiday['bg_offset_y'] : '';

				$text_h_align  = isset( $holiday['text_h_align'] ) ? $holiday['text_h_align'] : 'center';
				$text_v_align  = isset( $holiday['text_v_align'] ) ? $holiday['text_v_align'] : 'middle';
				$text_offset_x = isset( $holiday['text_offset_x'] ) ? $holiday['text_offset_x'] : '';
				$text_offset_y = isset( $holiday['text_offset_y'] ) ? $holiday['text_offset_y'] : '';

				$allowed_h = array( 'left', 'center', 'right' );
				if ( ! in_array( $text_h_align, $allowed_h, true ) ) {
					$text_h_align = 'center';
				}

				$allowed_v = array( 'top', 'middle', 'bottom' );
				if ( ! in_array( $text_v_align, $allowed_v, true ) ) {
					$text_v_align = 'middle';
				}
				$font_family = isset( $holiday['font_family'] ) ? $holiday['font_family'] : '';
				$font_size   = isset( $holiday['font_size'] ) ? $holiday['font_size'] : '';
				$font_style  = isset( $holiday['font_style'] ) ? $holiday['font_style'] : 'normal';
				$font_color  = isset( $holiday['font_color'] ) ? $holiday['font_color'] : '';

                $data = array(
					'id'            => $id,
					'label'         => $label,
					'message'       => $message,
					'effect'        => $effect,
					'location'      => $location,
					'bg_color'      => $bg_color,
					'bg_image'      => $bg_image,
					'bg_repeat'     => $bg_repeat,
					'bg_position'   => $bg_position,
					'bg_size'       => $bg_size,
					'banner_height' => $banner_height,
					'bg_opacity'    => $bg_opacity,
					'overlap'       => $overlap,
					'bg_offset_x'   => $bg_offset_x,
					'bg_offset_y'   => $bg_offset_y,
					'text_h_align'  => $text_h_align,
					'text_v_align'  => $text_v_align,
					'text_offset_x' => $text_offset_x,
					'text_offset_y' => $text_offset_y,
					'font_family'   => $font_family,
					'font_size'     => $font_size,
					'font_style'    => $font_style,
					'font_color'    => $font_color,
					'effect_groups' => ( isset( $holiday['effect_groups'] ) && is_array( $holiday['effect_groups'] ) ) ? $holiday['effect_groups'] : array(),

				);


                return apply_filters( 'ahs_today_holiday', $data, $holiday );
            }

            return null;
        }

        public function enqueue_assets() {
            if ( is_admin() ) {
                return;
            }

            if ( ! $this->today_holiday ) {
                return;
            }

            $style_handle  = 'ahs-holiday-effects';
            $script_handle = 'ahs-holiday-effects';

            wp_register_style(
                $style_handle,
                plugins_url( 'assets/celeb-right.css', __FILE__ ),
                array(),
                '3.0.0'
            );

            wp_register_script(
                $script_handle,
                plugins_url( 'assets/celeb-right.js', __FILE__ ),
                array(),
                '3.0.0',
                true
            );

            wp_enqueue_style( $style_handle );
            wp_enqueue_script( $script_handle );

            $settings = $this->settings;
            $effect   = $this->today_holiday['effect'];

            if ( ! empty( $settings['banner_style'] ) && 'auto' !== $settings['banner_style'] ) {
                if ( 'none' !== $settings['banner_style'] ) {
                    $effect = $settings['banner_style'];
                } else {
                    $effect = 'none';
                }
            }

            $default_leaf1 = plugins_url( 'assets/leaf1.svg', __FILE__ );
			$default_leaf2 = plugins_url( 'assets/leaf2.svg', __FILE__ );
			$default_leaf3 = plugins_url( 'assets/leaf3.svg', __FILE__ );
			$default_flag  = plugins_url( 'assets/us-flag.svg', __FILE__ );

			$effect_groups = $this->resolve_effect_groups_for_today( $settings, $this->today_holiday );

			wp_localize_script(
				$script_handle,
				'ahsHolidayData',
				array(
					'holidayId'     => $this->today_holiday['id'],
					'holidayLabel'  => $this->today_holiday['label'],
					'location'      => $this->today_holiday['location'],
					'effect_groups' => $effect_groups,
					'assets_defaults' => array(
						'leaves' => array( $default_leaf1, $default_leaf2, $default_leaf3 ),
						'flag'   => array( $default_flag ),
					),
				)
			);
        }

        public function render_banner() {
            if ( is_admin() ) {
                return;
            }

            if ( $this->banner_rendered ) {
                return;
            }

            if ( ! $this->today_holiday ) {
                return;
            }

            $settings = $this->get_settings();
            $effect   = $this->today_holiday['effect'];
            $location = $this->today_holiday['location'];
            $hook = current_action();
            if ( 'top' === $location || 'modal' === $location ) {
                if ( 'wp_body_open' !== $hook ) {
                    return;
                }
            } elseif ( 'bottom' === $location ) {
                if ( 'wp_footer' !== $hook ) {
                    return;
                }
            }

            if ( ! empty( $settings['banner_style'] ) && 'auto' !== $settings['banner_style'] ) {
                if ( 'none' !== $settings['banner_style'] ) {
                    $effect = $settings['banner_style'];
                } else {
                    $effect = 'none';
                }
            }

            $label   = esc_html( $this->today_holiday['label'] );
            $message = esc_html( $this->today_holiday['message'] );
            $note    = ! empty( $settings['global_note'] ) ? wp_kses_post( $settings['global_note'] ) : '';
            $bg_color      = ! empty( $this->today_holiday['bg_color'] ) ? $this->today_holiday['bg_color'] : '';
            $bg_image      = ! empty( $this->today_holiday['bg_image'] ) ? $this->today_holiday['bg_image'] : '';
            $bg_repeat     = ! empty( $this->today_holiday['bg_repeat'] ) ? $this->today_holiday['bg_repeat'] : '';
            $bg_position   = ! empty( $this->today_holiday['bg_position'] ) ? $this->today_holiday['bg_position'] : '';
            $bg_size       = ! empty( $this->today_holiday['bg_size'] ) ? $this->today_holiday['bg_size'] : '';
            $banner_height = ! empty( $this->today_holiday['banner_height'] ) ? $this->today_holiday['banner_height'] : '';
			$bg_opacity = isset( $this->today_holiday['bg_opacity'] ) ? $this->today_holiday['bg_opacity'] : '';
			$overlap    = isset( $this->today_holiday['overlap'] ) ? $this->today_holiday['overlap'] : '';
			$bg_offset_x   = isset( $this->today_holiday['bg_offset_x'] ) ? $this->today_holiday['bg_offset_x'] : '';
			$bg_offset_y   = isset( $this->today_holiday['bg_offset_y'] ) ? $this->today_holiday['bg_offset_y'] : '';
			$text_h_align  = ! empty( $this->today_holiday['text_h_align'] ) ? $this->today_holiday['text_h_align'] : 'center';
			$text_v_align  = ! empty( $this->today_holiday['text_v_align'] ) ? $this->today_holiday['text_v_align'] : 'middle';
			$text_offset_x = isset( $this->today_holiday['text_offset_x'] ) ? $this->today_holiday['text_offset_x'] : '';
			$text_offset_y = isset( $this->today_holiday['text_offset_y'] ) ? $this->today_holiday['text_offset_y'] : '';
            $font_family   = ! empty( $this->today_holiday['font_family'] ) ? $this->today_holiday['font_family'] : '';
            $font_size     = ! empty( $this->today_holiday['font_size'] ) ? intval( $this->today_holiday['font_size'] ) : 0;
            $font_style    = ! empty( $this->today_holiday['font_style'] ) ? $this->today_holiday['font_style'] : 'normal';
            $font_color    = ! empty( $this->today_holiday['font_color'] ) ? $this->today_holiday['font_color'] : '';
			$text_h_align  = in_array( $text_h_align, array( 'left', 'center', 'right' ), true ) ? $text_h_align : 'center';
			$text_v_align  = in_array( $text_v_align, array( 'top', 'middle', 'bottom' ), true ) ? $text_v_align : 'middle';
			$offset_x = ( '' !== $text_offset_x ) ? intval( $text_offset_x ) : 0;
			$offset_y = ( '' !== $text_offset_y ) ? intval( $text_offset_y ) : 0;
			$text_h_class = 'ahs-text-h-' . $text_h_align;
			$text_v_class = 'ahs-text-v-' . $text_v_align;
            $text_style_parts = array();
            if ( $font_color ) {
                $text_style_parts[] = 'color:' . $font_color;
            }
            if ( $font_family ) {
                $text_style_parts[] = 'font-family:' . $font_family;
            }
            if ( $font_size > 0 ) {
                $text_style_parts[] = 'font-size:' . $font_size . 'px';
            }

            $font_style = in_array( $font_style, array( 'normal', 'bold', 'italic', 'bold_italic' ), true ) ? $font_style : 'normal';

            if ( 'bold' === $font_style ) {
                $text_style_parts[] = 'font-weight:bold';
            } elseif ( 'italic' === $font_style ) {
                $text_style_parts[] = 'font-style:italic';
            } elseif ( 'bold_italic' === $font_style ) {
                $text_style_parts[] = 'font-weight:bold';
                $text_style_parts[] = 'font-style:italic';
            }

			$inner_style_parts = array();
			if ( 0 !== $offset_x || 0 !== $offset_y ) {
				$inner_style_parts[] = 'transform:translate(' . $offset_x . 'px,' . $offset_y . 'px)';
			}
            $inner_style_parts = array_merge( $inner_style_parts, $text_style_parts );

			$inner_style_attr = '';
			if ( ! empty( $inner_style_parts ) ) {
				$inner_style_attr = ' style="' . esc_attr( implode( ';', $inner_style_parts ) ) . '"';
			}

            $text_style_attr = '';
            if ( ! empty( $text_style_parts ) ) {
                $text_style_attr = ' style="' . esc_attr( implode( ';', $text_style_parts ) ) . '"';
            }
			
            $style_parts = array();

            if ( $bg_color ) {
                $style_parts[] = 'background-color:' . $bg_color;
            }
            if ( $bg_image ) {
                $style_parts[] = 'background-image:url(' . esc_url( $bg_image ) . ')';
            }
            if ( $bg_repeat ) {
                $style_parts[] = 'background-repeat:' . $bg_repeat;
            }
			$offset_x_int = ( '' !== $bg_offset_x ) ? intval( $bg_offset_x ) : 0;
			$offset_y_int = ( '' !== $bg_offset_y ) ? intval( $bg_offset_y ) : 0;

			$x_keyword = 'center';
			$y_keyword = 'top';

			if ( $bg_position ) {
				$pos_parts = explode( ' ', $bg_position );
				if ( isset( $pos_parts[0] ) ) {
					$x_keyword = $pos_parts[0];
				}
				if ( isset( $pos_parts[1] ) ) {
					$y_keyword = $pos_parts[1];
				}
			}

			$x_map = array(
				'left'   => 0,
				'center' => 50,
				'right'  => 100,
			);
			$y_map = array(
				'top'    => 0,
				'center' => 50,
				'bottom' => 100,
			);

			$x_pct = isset( $x_map[ $x_keyword ] ) ? $x_map[ $x_keyword ] : 50;
			$y_pct = isset( $y_map[ $y_keyword ] ) ? $y_map[ $y_keyword ] : 0;

			$pos_x_expr = $x_pct . '%';
			$pos_y_expr = $y_pct . '%';

			if ( 0 !== $offset_x_int ) {
				$pos_x_expr = 'calc(' . $x_pct . '% + ' . $offset_x_int . 'px)';
			}
			if ( 0 !== $offset_y_int ) {
				$pos_y_expr = 'calc(' . $y_pct . '% + ' . $offset_y_int . 'px)';
			}

			$style_parts[] = 'background-position:' . $pos_x_expr . ' ' . $pos_y_expr;

            if ( $bg_size ) {
                $style_parts[] = 'background-size:' . $bg_size;
            }
            if ( $banner_height ) {
                $style_parts[] = 'min-height:' . intval( $banner_height ) . 'px';
            }
			
			$overlap_int = ( '' !== $overlap ) ? intval( $overlap ) : 0;
			if ( 0 !== $overlap_int ) {
				$style_parts[] = 'margin-bottom:' . ( 0 - $overlap_int ) . 'px';
			}
			
		if ( '' !== $bg_opacity ) {
			$bg_opacity_int = intval( $bg_opacity );
			if ( $bg_opacity_int < 0 ) {
				$bg_opacity_int = 0;
			} elseif ( $bg_opacity_int > 100 ) {
				$bg_opacity_int = 100;
			}
			$opacity = 1 - ( $bg_opacity_int / 100 );
			$style_parts[] = 'opacity:' . $opacity;
		}


            $bg_attr = '';
            if ( ! empty( $style_parts ) ) {
                $bg_attr = ' style="' . esc_attr( implode( ';', $style_parts ) ) . '"';
            }

            if ( 'modal' === $location ) : ?>
                <div class="ahs-modal-overlay" id="ahs-holiday-modal">
                    <div class="ahs-modal-content"<?php echo $bg_attr; ?>>
                        <button type="button" class="ahs-modal-close">&times;</button>
                        <div class="ahs-banner-title"<?php echo $text_style_attr; ?>><?php echo $label; ?></div>
                        <?php if ( $message ) : ?>
                            <div class="ahs-banner-message"<?php echo $text_style_attr; ?>><?php echo $message; ?></div>
                        <?php endif; ?>
                        <?php if ( $note ) : ?>
                            <div class="ahs-banner-note"<?php echo $text_style_attr; ?>><?php echo $note; ?></div>
                        <?php endif; ?>

                    </div>
                </div>
            <?php
            else :
                $pos_class = ( 'bottom' === $location ) ? 'ahs-banner-bottom' : 'ahs-banner-top';
					?>
					<div class="ahs-banner <?php echo esc_attr( $pos_class . ' ' . $text_h_class . ' ' . $text_v_class ); ?> ahs-banner-effect-<?php echo esc_attr( $effect ); ?>"<?php echo $bg_attr; ?>>
						<div class="ahs-banner-inner"<?php echo $inner_style_attr; ?>>
							<div class="ahs-banner-title"><?php echo $label; ?></div>
							<?php if ( $message ) : ?>
								<div class="ahs-banner-message"><?php echo $message; ?></div>
							<?php endif; ?>
							<?php if ( $note ) : ?>
								<div class="ahs-banner-note"><?php echo $note; ?></div>
							<?php endif; ?>
						</div>
					</div>
					<?php

            endif;

            $render_groups = $this->resolve_effect_groups_for_today( $settings, $this->today_holiday );
			$has_effects = false;

			if ( is_array( $render_groups ) ) {
				foreach ( $render_groups as $g ) {
					$t = isset( $g['type'] ) ? $g['type'] : 'none';
					if ( $t !== 'none' ) { $has_effects = true; break; }
				}
			}

			if ( $has_effects ) : ?>
				<div id="ahs-effect-container" class="ahs-effect-container ahs-effect-<?php echo esc_attr( $effect ); ?> ahs-effect-multi"></div>
            <?php
            endif;

            $this->banner_rendered = true;
        }
    }

    AHS_Holiday_Effects::get_instance();

function ahs_flush_all_caches() {
    if ( function_exists( 'wp_cache_flush' ) ) {
        wp_cache_flush();
    }
    do_action( 'wp_super_cache_clear_cache' );
    if ( function_exists( 'wp_cache_clear_cache' ) ) {
        wp_cache_clear_cache( 0 );
    }
}

function ahs_schedule_daily_cache_flush() {
    if ( ! wp_next_scheduled( 'ahs_daily_cache_flush' ) ) {
        $timestamp = strtotime( 'tomorrow 00:05:00' );
        if ( $timestamp ) {
            wp_schedule_event( $timestamp, 'daily', 'ahs_daily_cache_flush' );
        }
    }
}

function ahs_clear_daily_cache_flush() {
    $timestamp = wp_next_scheduled( 'ahs_daily_cache_flush' );
    if ( $timestamp ) {
        wp_unschedule_event( $timestamp, 'ahs_daily_cache_flush' );
    }
}

add_action( 'ahs_daily_cache_flush', 'ahs_flush_all_caches' );
register_activation_hook( __FILE__, 'ahs_schedule_daily_cache_flush' );
register_deactivation_hook( __FILE__, 'ahs_clear_daily_cache_flush' );

}
